<?php

namespace EAddonsProFormActions\Modules\Actions\Actions;

use EAddonsForElementor\Base\Base_Action;
use Elementor\Controls_Manager;
use EAddonsForElementor\Core\Utils;
use EAddonsForElementor\Core\Utils\Form;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

class Sms_Aruba extends Base_Action {

    /**
     * Get Name
     *
     * Return the action name
     *
     * @access public
     * @return string
     */
    public function get_name() {
        return 'sms-aruba';
    }

    /**
     * Get Label
     *
     * Returns the action label
     *
     * @access public
     * @return string
     */
    public function get_label() {
        return esc_html__('Aruba SMS', 'e-addons');
    }

    /*
      public function get_icon() {
      return 'eadd-el-form-pro-act-send-telegtam';
      }

      public function get_pid() {
      return 1257;
      }
     */

    /**
     * Register Settings Section
     *
     * Registers the Action controls
     *
     * @access public
     * @param \Elementor\Widget_Base $widget
     */
    public function register_settings_section($widget) {

        $this->start_controls_section($widget);

        $widget->add_control(
                'e_form_sms_aruba_user', [
            'label' => esc_html__('Aruba SMS Username', 'e-addons'),
            'type' => \Elementor\Controls_Manager::TEXT,
            'placeholder' => 'Sms12345',
                ]
        );

        $widget->add_control(
                'e_form_sms_aruba_pass', [
            'label' => esc_html__('Aruba SMS Password', 'e-addons'),
            'type' => \Elementor\Controls_Manager::TEXT,
            'separator' => 'after',
                ]
        );
        
        $widget->add_control(
                'e_form_sms_aruba_user_key', [
            'label' => esc_html__('Aruba SMS User Key', 'e-addons'),
            'type' => \Elementor\Controls_Manager::TEXT,
            'condition' => [
                'e_form_sms_aruba_user' => '',
            ],
                ]
        );

        $widget->add_control(
                'e_form_sms_aruba_access_token', [
            'label' => esc_html__('Aruba SMS Token', 'e-addons'),
            'type' => \Elementor\Controls_Manager::TEXT,
            'separator' => 'after',
            'condition' => [
                'e_form_sms_aruba_user' => '',
            ],
                ]
        );

        
        $widget->add_control(
                'e_form_sms_aruba_sender', [
            'label' => esc_html__('Sender', 'e-addons'),
            'type' => \Elementor\Controls_Manager::TEXT,
            'placeholder' => __('Sender Name', 'e-addons'),
            'default' => __('[field id="name"]'),
            'description' => __('The Sender name. If the message type allows a custom TPOA and the field is left empty, the user’s preferred TPOA is used. Must be empty if the message type does not allow a custom TPOA'),                    
                ]
        );
        $widget->add_control(
                'e_form_sms_aruba_recipient', [
            'label' => esc_html__('Recipient', 'e-addons'),
            'type' => \Elementor\Controls_Manager::TEXT,
            'placeholder' => __('+349123456789, 00393471234567', 'e-addons'),
            'description' => __('A list of recipents phone numbers. The recipients numbers must be in the international format'),
                ]
        );
        $widget->add_control(
                'e_form_sms_aruba_message', [
            'label' => esc_html__('Message', 'e-addons'),
            'type' => \Elementor\Controls_Manager::TEXTAREA,
            'default' => __('[field id="message"]'),
            'description' => __('The body of the message. *Message max length could be 160 chars when using low-quality SMSs. New line char needs to be codified as “\n”.'),
            'separator' => 'after',
                ]
        );
        
        $widget->add_control(
                'e_form_sms_aruba_quality', [
            'label' => esc_html__('Message Quality', 'e-addons'),
            'type' => \Elementor\Controls_Manager::SELECT,
            'options' => [
                'N' => __('High'),
                'L' => __('Medium'),
            ],
            'default' => 'N',
                ]
        );
        $widget->add_control(
                'e_form_sms_aruba_encoding', [
            'label' => esc_html__('Encoding', 'e-addons'),
            'type' => \Elementor\Controls_Manager::SELECT,
            'options' => [
                'gsm' => __('gsm'),
                'ucs2' => __('ucs2'),
            ],
            'default' => 'gsm',
            'description' => __('The SMS encoding. Use UCS2 for non standard character sets'),
                ]
        );
        
        $widget->add_control(
                'e_form_sms_aruba_max_fragments', [
            'label' => esc_html__('Max fragments', 'e-addons'),
            'type' => \Elementor\Controls_Manager::NUMBER,
            'min' => 1,
            'description' => __('The number of maximum fragments allowed per sms. Configured by user in settings page, Default 7'),
                ]
        );
        

        Utils::add_help_control($this, $widget);

        $widget->end_controls_section();
    }

    /**
     * Run
     *
     * Runs the action after submit
     *
     * @access public
     * @param \ElementorPro\Modules\Forms\Classes\Form_Record $record
     * @param \ElementorPro\Modules\Forms\Classes\Ajax_Handler $ajax_handler
     */
    public function run($record, $ajax_handler) {

        $fields = Form::get_form_data($record);
        $settings = $this->get_settings(true, $fields);

        if (!defined('SMS_ARUBA_BASEURL')) {
            define("SMS_ARUBA_BASEURL", "https://smspanel.aruba.it/API/v1.0/REST/");
        }
                
        $recipient = Utils::explode($settings['e_form_sms_aruba_recipient']);
        
        if (!empty($recipient)) {

            foreach ($recipient as $phone) {
                if (substr($phone, 0, 1) != '+' && substr($phone, 0, 2) != '00') {
                    $ajax_handler->add_error_message(__('The recipients numbers must be in the international format', 'e-addons'));
                    return;
                }
            }

            $auth = false;
            if (!empty($settings['e_form_sms_aruba_user']) && !empty($settings['e_form_sms_aruba_pass'])) {
                $auth = $this->login($settings['e_form_sms_aruba_user'], $settings['e_form_sms_aruba_pass']);
                if (!$auth) {                
                    $ajax_handler->add_error_message(__('Aruba SMS Authentication credential are not valid', 'e-addons'));
                    return;
                }
            }
            if (!$auth && !empty($settings['e_form_sms_aruba_user_key']) && !empty($settings['e_form_sms_aruba_access_token'])) {
                $auth = array(
                    $settings['e_form_sms_aruba_user_key'], 
                    $settings['e_form_sms_aruba_access_token']
                );
            }            
            if (!$auth) {                
                $ajax_handler->add_error_message(__('Aruba SMS Authentication credential needed', 'e-addons'));
                return;
            }

            $message = $settings['e_form_sms_aruba_message'];
            $message = Form::get_plain_txt($message);
            $message = apply_filters('elementor_pro/forms/wp_sms_message', $message);

            $settings_sms = array(
                "message" => $message,
                "message_type" => $settings['e_form_sms_aruba_quality'],
                //"returnCredits" => false,
                "recipient" => $recipient,
                "sender" => $settings['e_form_sms_aruba_sender'] ? $settings['e_form_sms_aruba_sender'] : null, // Place here a custom sender if desired
                //"scheduled_delivery_time" => date('YmdHi', strtotime("+5 minutes")), // postpone by 5 minutes
                "encoding" => $settings['e_form_sms_aruba_encoding'],
            );
                       
            if ($max_fragments = intval($settings['e_form_sms_aruba_max_fragments'])) {
                $settings_sms['max_fragments'] = $max_fragments;
            }

            //var_dump($settings_sms); die();
            $smsSent = $this->sendSMS($auth, $settings_sms);

            if ($smsSent->result != "OK") {
                $ajax_handler->add_error_message($smsSent->result);
                return;
            }

            /**
             * Elementor form sms sent.
             *
             * Fires when an sms was sent successfully.
             *
             * @since 1.0.0
             *
             * @param array       $settings Form settings.
             * @param Form_Record $record   An instance of the form record.
             */
            do_action('e_addons/forms/sms_sent', $settings_sms, $record);
        }
    }

    /**
     * Authenticates the user given it's username and password.
     * Returns the pair user_key, Session_key
     */
    public function login($username, $password) {
        $ch = curl_init();
        //var_dump(SMS_ARUBA_BASEURL . 'login?username=' . $username .'&password=' . $password); die();
        curl_setopt($ch, CURLOPT_URL, SMS_ARUBA_BASEURL .
                'login?username=' . $username .
                '&password=' . $password);

        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);

        $response = curl_exec($ch);
        $info = curl_getinfo($ch);
        curl_close($ch);

        if ($info['http_code'] != 200) {
            return null;
        }

        return explode(";", $response);
    }

    /**
     * Sends an SMS message
     */
    public function sendSMS($auth, $sendSMS) {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, SMS_ARUBA_BASEURL . 'sms');
        curl_setopt($ch, CURLOPT_HTTPHEADER, array(
            'Content-type: application/json',
            'user_key: ' . $auth[0],
            'Session_key: ' . $auth[1]
        ));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($sendSMS));
        $response = curl_exec($ch);
        $info = curl_getinfo($ch);
        curl_close($ch);

        if ($info['http_code'] != 201) {
            return null;
        }

        return json_decode($response);
    }

}
