<?php

namespace EAddonsProFormActions\Modules\Actions\Actions;

use EAddonsForElementor\Base\Base_Action;
use Elementor\Controls_Manager;
use EAddonsForElementor\Core\Utils;
use EAddonsForElementor\Core\Utils\Form;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

class Spreadsheet extends Base_Action {

    /**
     * Get Name
     *
     * Return the action name
     *
     * @access public
     * @return string
     */
    public function get_name() {
        return 'spreadsheet';
    }

    public function get_icon() {
        return 'eadd-el-form-pro-act-csv';
    }

    public function get_pid() {
        return 277;
    }

    /**
     * Get Label
     *
     * Returns the action label
     *
     * @access public
     * @return string
     */
    public function get_label() {
        return esc_html__('Spreadsheet', 'e-addons');
    }

    /**
     * Register Settings Section
     *
     * Registers the Action controls
     *
     * @access public
     * @param \Elementor\Widget_Base $widget
     */
    public function register_settings_section($widget) {

        $this->start_controls_section($widget);
        
        $widget->add_control(
                'e_form_spreadsheet_format', [
            'label' => esc_html__('File Format', 'e-addons'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'csv' => [
                    'title' => esc_html__('CSV', 'e-addons'),
                    'icon' => 'eicon-table',
                ],
                'xlsx' => [
                    'title' => esc_html__('XLSX', 'e-addons'),
                    'icon' => 'fa fa-file-excel-o',
                ],
                'xls' => [
                    'title' => esc_html__('XLS', 'e-addons'),
                    'icon' => 'fa fa-file-excel-o',
                ],
                'ods' => [
                    'title' => esc_html__('ODS', 'e-addons'),
                    'icon' => 'eicon-document-file',
                ],
            ],
            'toggle' => false,
            'default' => 'csv',
                ]
        );
        $widget->add_control(
                'e_form_spreadsheet_path', [
            'label' => esc_html__('File Path', 'e-addons'),
            'type' => Controls_Manager::TEXT,
            'default' => 'elementor/spreadsheets/spreadsheet_' . time(),
            'description' => esc_html__('The spreadsheets path, extension will be added automatically', 'e-addons'),
            'label_block' => true,
                ]
        );
        $widget->add_control(
                'e_form_spreadsheet_htaccess', [
            'label' => esc_html__('Prevent direct access', 'e-addons'),
            'type' => Controls_Manager::SWITCHER,
            'description' => esc_html__('Block folder direct access to file via public url', 'e-addons'),
                ]
        );
        

        $widget->add_control(
                'e_form_spreadsheet_header', [
            'label' => esc_html__('Add Header Row', 'e-addons'),
            'type' => Controls_Manager::SWITCHER,
            'default' => 'yes',
            'description' => esc_html__('Insert all the fields name as first row, it\'s necessary to identify the fields if you reorder them or add new ones.', 'e-addons'),
                ]
        );
        $widget->add_control(
                'e_form_spreadsheet_position', [
            'label' => esc_html__('Insert position', 'e-addons'),
            'type' => Controls_Manager::CHOOSE,
            'options' => [
                'first' => [
                    'title' => esc_html__('First', 'e-addons'),
                    'icon' => 'eicon-arrow-up',
                ],
                'last' => [
                    'title' => esc_html__('Last', 'e-addons'),
                    'icon' => 'eicon-arrow-down',
                ],
            ],
            'toggle' => false,
            'default' => 'last',
                ]
        );

        $widget->add_control(
                'e_form_csv_delimiter', [
            'label' => esc_html__('Delimiter', 'e-addons'),
            'type' => Controls_Manager::TEXT,
            'default' => ',',
            'condition' => [
                'e_form_spreadsheet_format' => 'csv',
            ],
                ]
        );
        $widget->add_control(
                'e_form_csv_enclosure', [
            'label' => esc_html__('Enclosure', 'e-addons'),
            'type' => Controls_Manager::TEXT,
            'default' => '',
            'condition' => [
                'e_form_spreadsheet_format' => 'csv',
            ],
                ]
        );



        Utils::add_help_control($this, $widget);

        $widget->end_controls_section();
    }

    /**
     * Run
     *
     * Runs the action after submit
     *
     * @access public
     * @param \ElementorPro\Modules\Forms\Classes\Form_Record $record
     * @param \ElementorPro\Modules\Forms\Classes\Ajax_Handler $ajax_handler
     */
    public function run($record, $ajax_handler) {
        $fields = Form::get_form_data($record, false);
        $settings = $this->get_settings(true, $fields);

        $upload = wp_upload_dir();
        $upload_dir = $upload['basedir'];
        $file_path = $upload_dir . DIRECTORY_SEPARATOR . $settings['e_form_spreadsheet_path'] . '.' . $settings['e_form_spreadsheet_format'];
        $file_dir = dirname($file_path);

        if ((!is_dir($file_dir) && !mkdir($file_dir, 0755, true))) {
            $ajax_handler->add_error_message('Error on SPREADSHEET folder creation in: ' . $file_dir);
        }

        $htaccess = $file_dir . DIRECTORY_SEPARATOR . '.htaccess';
        //var_dump($htaccess); die();
        $htblock = 'Options -Indexes' . PHP_EOL . '<files "*">' . PHP_EOL . 'order allow,deny' . PHP_EOL . 'deny from all' . PHP_EOL . '</files>';
        if (empty($settings['e_form_spreadsheet_htaccess'])) {
            if (is_file($htaccess)) {
                $htfile = file_get_contents($htaccess);
                if ($htfile == $htblock) {
                    if (\Elementor\Plugin::$instance->editor->is_edit_mode()) {
                        //echo '<div class="elementor-alert elementor-alert-danger"><h5 class="elementor-alert-title">Warning</h5>The folder is secured. The HTACCESS file will be removed.</div>';
                    } else {
                        //$htblock = 'RewriteEngine on'.PHP_EOL.'RedirectMatch 403 ^/folder/file.php$';
                        unlink($htaccess);
                    }
                }
            }
        } else {
            if (!is_file($htaccess)) {
                if (\Elementor\Plugin::$instance->editor->is_edit_mode()) {
                    //echo '<div class="elementor-alert elementor-alert-danger"><h5 class="elementor-alert-title">Warning</h5>The folder is not secured. An HTACCESS file will be generated.</div>';
                } else {
                    file_put_contents($htaccess, $htblock);
                }
            }
        }

        $format = ucfirst($settings['e_form_spreadsheet_format']);

        // remove unnecessary fields
        $unset = array();
        if (!empty($settings['form_fields'])) {
            foreach ($settings['form_fields'] as $key => $item) {
                if (in_array($item["field_type"], array('step', 'honeypot', 'recaptcha', 'recaptcha_v3', 'reset', 'submit'))) {
                    $unset[] = $item['custom_id'];
                }
            }
        }
        foreach ($fields as $key => $value) {
            if (in_array($key, $unset)) {
                unset($fields[$key]);
            }
        }

        // https://phpspreadsheet.readthedocs.io
        if (file_exists($file_path)) {
            $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReader($format);

            if ($settings['e_form_spreadsheet_format'] == 'csv') {
                $reader->setDelimiter($settings['e_form_csv_delimiter']);
                $reader->setEnclosure($settings['e_form_csv_enclosure']);
            }

            $spreadsheet = $reader->load($file_path);
            $sheet = $spreadsheet->getActiveSheet();
        } else {
            $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();
            if ($settings['e_form_spreadsheet_header']) {
                // add header            
                $fields_key = array_keys($fields);
                $sheet->fromArray($fields_key, NULL, 'A1');
            }
        }

        if ($settings['e_form_spreadsheet_header']) {
            // reorder fields by header key
            $sheet_array = $sheet->toArray();
            $header_key = reset($sheet_array);
            $reversed = array_reverse($header_key);
            foreach ($reversed as $key) {
                $value = '';
                if (isset($fields[$key])) {
                    $value = $fields[$key];
                    unset($fields[$key]);
                }
                array_unshift($fields, $value);
            }
            // update header row with new fields
            $i = 1;
            foreach ($fields as $key => $value) {
                if ($i > count($header_key)) {
                    $header_key[] = $key;
                }
                $i++;
            }
            $sheet->fromArray($header_key, NULL, 'A1');
        }

        // add new row
        if ($settings['e_form_spreadsheet_position'] == 'first') {
            $row = $settings['e_form_spreadsheet_header'] ? 2 : 1; // first before heading             
            $sheet->insertNewRowBefore($row);
        } else {
            $row = $sheet->getHighestRow() + 1; // last
        }
        $sheet->fromArray($fields, NULL, 'A' . $row);

        $writer = \PhpOffice\PhpSpreadsheet\IOFactory::createWriter($spreadsheet, $format);
        if ($settings['e_form_spreadsheet_format'] == 'csv') {
            $writer->setDelimiter($settings['e_form_csv_delimiter']);
            $writer->setEnclosure($settings['e_form_csv_enclosure']);
        }
        $writer->save($file_path);
    }

}
