<?php
/**
 * =======================================
 * MainWP Database Updater Updates Plugins
 * =======================================
 */
namespace MainWP\Extensions\DatabaseUpdater;

class MainWP_Database_Updater_Updates_Plugins {

    const MAINWP_VIEW_PER_SITE         = 1;
    const MAINWP_VIEW_PER_PLUGIN_THEME = 0;
    const MAINWP_VIEW_PER_GROUP        = 2;

    // Singleton
    private static $instance = null;

    // Singleton
    public static $supported_plugins = array(
        'woocommerce/woocommerce.php',
        'elementor/elementor.php',
        'elementor-pro/elementor-pro.php',
    );

    /**
     *  User can update plugins.
     *
     * @var bool $user_can_update_plugins User can update plugins.
     */
    public static $user_can_update_plugins = null;

    /**
     * Get Instance
     *
     * Creates public static instance.
     *
     * @static
     *
     * @return the class.
     */
    static function get_instance() {
        if ( null == self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * constructor.
     *
     * Run each time the class is called.
     */
    public function __construct() {
    }


    /**
     * Method get_supported_plugins().
     *
     * Get supported plugins.
     *
     * @static
     */
    public static function get_supported_plugins() {
        return self::$supported_plugins;
    }

    /**
     * Renders updates page.
     */
    public function render_db_updates( $website = false ) { // phpcs:ignore Generic.Metrics.CyclomaticComplexity -- current complexity is the only way to achieve desired results, pull request solutions appreciated.

        $individual = false;
        if ( ! empty( $website ) ) {
            $websites   = static::get_sites( $website->id );
            $individual = true;
        } else {
            $websites = static::get_sites();
        }

        $site_view = static::MAINWP_VIEW_PER_PLUGIN_THEME;

        $userExtension = apply_filters( 'mainwp_get_user_extension', false );

        $total_plugin_db_upgrades = 0;

        $allPlugins  = array();
        $pluginsInfo = array();

        MainWP_Database_Updater_Utility::data_seek( $websites );
        while ( $websites && ( $website = MainWP_Database_Updater_Utility::fetch_object( $websites ) ) ) {
            $plugin_db_upgrades = ! empty( $website->plugin_db_upgrades ) ? json_decode( $website->plugin_db_upgrades, true ) : array();
            if ( $website->is_ignorePluginUpdates ) {
                $plugin_db_upgrades = array();
            }

            if ( is_array( $plugin_db_upgrades ) && ! $website->is_ignorePluginUpdates ) {
                $_ignored_plugins = ! empty( $website->ignored_plugins ) ? json_decode( $website->ignored_plugins, true ) : array();
                if ( is_array( $_ignored_plugins ) ) {
                    $plugin_db_upgrades = array_diff_key( $plugin_db_upgrades, $_ignored_plugins );
                }

                $_ignored_plugins = ! empty( $userExtension->ignored_plugins ) ? json_decode( $userExtension->ignored_plugins, true ) : array();
                if ( is_array( $_ignored_plugins ) ) {
                    $plugin_db_upgrades = array_diff_key( $plugin_db_upgrades, $_ignored_plugins );
                }

                // supported the WC plugin.
                if ( is_array( $plugin_db_upgrades ) ) {
                    foreach ( self::$supported_plugins as $supp_slug ) {
                        if ( isset( $plugin_db_upgrades[ $supp_slug ] ) ) {
                            ++$total_plugin_db_upgrades;
                        }
                    }
                }
            }

            if ( self::MAINWP_VIEW_PER_PLUGIN_THEME == $site_view ) {

                if ( is_array( $plugin_db_upgrades ) ) {
                    foreach ( $plugin_db_upgrades as $slug => $plugin_db_upgrade ) {
                        if ( ! in_array( $slug, self::$supported_plugins ) ) {
                            continue;
                        }
                        if ( ! isset( $allPlugins[ $slug ] ) ) {
                            $allPlugins[ $slug ] = array(
                                'name' => esc_html( $plugin_db_upgrade['Name'] ),
                                'cnt'  => 1,
                            );
                        } else {
                            ++$allPlugins[ $slug ]['cnt'];
                        }

                        $pluginsInfo[ $slug ] = array(
                            'name'   => esc_html( $plugin_db_upgrade['Name'] ),
                            'slug'   => isset( $plugin_db_upgrade['update']['slug'] ) ? esc_html( $plugin_db_upgrade['update']['slug'] ) : '',
                            'active' => true,
                        );
                    }
                }
            }
        }

        $allPluginsSortBy = apply_filters( 'mainwp_updates_plugins_sort_by', 'name' );

        MainWP_Database_Updater_Utility::array_sort( $allPlugins, $allPluginsSortBy );

        self::render_plugins_update_tab( $websites, $total_plugin_db_upgrades, $userExtension, $allPlugins, $pluginsInfo, $site_view, $individual );
        self::render_js_updates( $site_view );
        self::render_updates_modal();

        MainWP_Database_Updater_Utility::free_result( $websites );
    }


    /**
     * Renders JavaScript for update page.
     *
     * @param string $site_view current site view.
     */
    public static function render_js_updates( $site_view ) {
        $table_features = array(
            'searching' => 'false',
            'paging'    => 'false',
            'stateSave' => 'true',
            'info'      => 'false',
            'exclusive' => 'false',
            'duration'  => '200',
        );
        /**
         * Filter: mainwp_updates_table_features
         *
         * Filters the Updates table features.
         *
         * @since 4.1
         */
        $table_features = apply_filters( 'mainwp_updates_table_features', $table_features );
        ?>
        <script type="text/javascript">
            let parentTable = null;
            jQuery( document ).ready( function ($) {
                jQuery( '.mainwp-db-updater-table .ui.accordion' ).accordion( {
                    "exclusive": <?php echo $table_features['exclusive']; ?>,
                    "duration": <?php echo $table_features['duration']; ?>
                } );

                parentTable = $('#mainwp-db-plugins-updates-table').DataTable( {
                    "searching": true,
                    "paging" : false,
                    "info" : true,
                    "ordering": false,
                    "columnDefs" : [
                        { "orderable": false, "targets": "no-sort" },
                    ],
                    "language" : { "emptyTable": "No available updates. Please sync your MainWP Dashboard with Child Sites to see if there are any new updates available." }
                } );

                $( '.mainwp-db-updater-table .ui.accordion tr.title' ).each( function( i ) {
                    mainwp_database_updater_showhide_child_row_tbl( this );
                    // double clicks to make it deactive by default.
                    $( this ).trigger( 'click' );
                    $( this ).trigger( 'click' );
                    $(this).next('tr').addClass('child-checkbox'); // to support check all.
                });

            } );

            mainwp_datatable_init_and_fix_recalc = function(selector){
                jQuery(selector).each( function(e) {
                    if(jQuery(this).is(":visible")){
                        console.log('visible ' + jQuery(this).attr('id'));
                        jQuery(this).css( 'display', 'table' );
                        jQuery(this).DataTable().destroy();
                        var tb = jQuery(this).DataTable({
                            "paging":false,
                            "info": false,
                            "searching": false,
                            "ordering" : false,
                            "responsive": true,
                        });
                        tb.columns.adjust();
                        tb.responsive.recalc();
                    }
                });
            }

            jQuery( document ).on( 'click', '.trigger-all-accordion', function() {
                if ( jQuery( this ).hasClass( 'active' ) ) {
                    jQuery( this ).removeClass( 'active' );
                    jQuery( '.mainwp-db-updater-table .ui.accordion tr.title' ).each( function( i ) {
                        if ( jQuery( this ).hasClass( 'active' ) ) {
                            jQuery( this ).trigger( 'click' ); // to deactive.
                        }
                    } );
                } else {
                    jQuery( this ).addClass( 'active' );
                    jQuery( '.mainwp-db-updater-table .ui.accordion tr.title' ).each( function( i ) {
                        if ( !jQuery( this ).hasClass( 'active' ) ) {
                            jQuery( this ).trigger( 'click' ); // to active.
                        }
                    } );
                }
            } );

            let mainwp_database_updater_showhide_child_row_tbl = function( obj ){
                console.log('Show|hide child table.');
                if(!parentTable){
                    return;
                }
                const tr = jQuery(obj).closest('tr');
                const row = parentTable.row(tr);
                const rowIndex = row.index();
                if (!row.child.isShown()) {
                    // Show the nested table placeholder
                    if(jQuery('#mainwp-plugins-db-updates-childrow-holder-' + rowIndex + ' > table').length){
                        let child = jQuery('#mainwp-plugins-db-updates-childrow-holder-' + rowIndex + ' > table')[0];
                        row.child(jQuery(child).clone()).show();
                        tr.addClass('shown');
                        // Now initialize the nested DataTable
                        jQuery('#mainwp-db-plugins-updates-table #mainwp-plugins-db-updates-childrow-' + rowIndex ).DataTable({
                            "paging":false,
                            "info": false,
                            "searching": false,
                            "ordering" : false,
                            "responsive": true,
                        });
                        jQuery('#mainwp-db-plugins-updates-table #mainwp-plugins-db-updates-childrow-' + rowIndex + ' .ui.checkbox').checkbox();
                    }
                }
            }
        </script>
        <?php
    }


    /**
     * Displays the updates modal window during updates.
     */
    public static function render_updates_modal() {
        ?>
        <div class="ui modal" id="updatesoverview-backup-box">
            <div class="header"><?php esc_html_e( 'Backup Check', 'mainwp-database-updater-extension' ); ?></div>
            <div class="scrolling content mainwp-modal-content"></div>
            <div class="actions mainwp-modal-actions">
                <input id="updatesoverview-backup-all" type="button" name="Backup All" value="<?php esc_html_e( 'Backup All', 'mainwp-database-updater-extension' ); ?>" class="button-primary"/>
                <a id="updatesoverview-backup-now" href="javascript:void(0)" target="_blank" style="display: none"  class="button-primary button"><?php esc_html_e( 'Backup Now', 'mainwp-database-updater-extension' ); ?></a>&nbsp;
                <input id="updatesoverview-backup-ignore" type="button" name="Ignore" value="<?php esc_html_e( 'Ignore', 'mainwp-database-updater-extension' ); ?>" class="button"/>
            </div>
        </div>
        <?php
    }


    /**
     * Gets sites for updates
     *
     * @return object Object containing websites info.
     */
    public static function get_sites( $site_id = false ) {
        $params = array(
            'extra_select_wp_fields' => array( 'plugins', 'plugin_db_upgrades', 'is_ignorePluginUpdates', 'ignored_plugins' ),
            'extra_view'             => array( 'premium_upgrades', 'favi_icon', 'plugin_db_upgrades' ),
        );

        if ( $site_id ) {
            $params['extra_where'] = ' wp.id = ' . (int) $site_id . ' ';
        }

        $view = get_user_option( 'mainwp_staging_options_updates_view' );
        if ( empty( $site_id ) ) {
            $params['is_staging'] = 'staging' === $view ? 'yes' : 'no';
        }

        $sql = MainWP_Database_Updater_Utility::get_sql_websites_for_current_user( $params );
        return MainWP_Database_Updater_Utility::db_query( $sql );
    }

    /**
     * Renders WP updates tab.
     *
     * @param object $websites               Object containing child sites info.
     * @param int    $total_plugin_db_upgrades  Number of available db plugin updates.
     * @param object $userExtension          User extension.
     * @param array  $allPlugins             Array of all plugins.
     * @param array  $pluginsInfo            Array of all plugins info.
     * @param string $site_view              Current view.
     * @param bool   $individual             Individual.
     */
    public static function render_plugins_update_tab( $websites, $total_plugin_db_upgrades, $userExtension, $allPlugins, $pluginsInfo, $site_view, $individual = false ) {

        $trustedPlugins = ! empty( $userExtension->trusted_plugins ) ? json_decode( $userExtension->trusted_plugins, true ) : array();

        if ( ! is_array( $trustedPlugins ) ) {
            $trustedPlugins = array();
        }
        ?>
        <div class="ui active tab mainwp-db-updater-table" data-tab="plugins-updates" id="mainwp-manage-updates">
        <?php
        if ( $individual ) {
            static::render_individual_plugins_updates( $websites, $total_plugin_db_upgrades, $userExtension, $allPlugins, $pluginsInfo, $trustedPlugins, $individual );
        } else {
            static::render_plugins_updates( $websites, $total_plugin_db_upgrades, $userExtension, $allPlugins, $pluginsInfo, $trustedPlugins );
        }
        ?>
        </div>
        <?php
    }

    /**
     * Method render_plugins_updates()
     *
     * Render Plugins updates
     *
     * @param object $websites the websites.
     * @param int    $total_plugin_db_upgrades  Number of available db plugin updates.
     * @param mixed  $userExtension user extension.
     * @param array  $allPlugins all plugins.
     * @param array  $pluginsInfo pugins information.
     * @param array  $trustedPlugins trusted plugins.
     */
	public static function render_plugins_updates( $websites, $total_plugin_db_upgrades, $userExtension, $allPlugins, $pluginsInfo, $trustedPlugins ) { // phpcs:ignore -- not quite complex method.
        $child_rows = array();
        ?>
        <?php if ( MainWP_Database_Updater_Utility::show_mainwp_message( 'mainwp-db-updater-info' ) ) : ?>
            <div class="ui info message">
                <i class="close icon mainwp-notice-dismiss" notice-id="mainwp-db-updater-info"></i>
                <?php printf( __( 'MainWP Database Updater Extension allows you to trigger detected Database updates for supported plugins. For more information, review %1$shelp documentation%2$s.', 'mainwp-database-updated-extension' ), '<a href="https://mainwp.com/kb/mainwp-database-updater-extension/" target="_blank">', '</a>' ); ?>
            </div>
        <?php endif; ?>
        <table class="ui tablet stackable table mainwp-manage-updates-table master-checkbox not-default-init" id="mainwp-db-plugins-updates-table" style="width: 100% !important;">
            <thead>
                <tr>
                    <th class="collapsing no-sort trigger-all-accordion"><span class="trigger-handle-arrow"><i class="caret right icon"></i><i class="caret down icon"></i></span></th>
                    <th class="indicator-accordion-sorting handle-accordion-sorting">
                    <?php esc_html_e( 'Plugin', 'mainwp-database-updater-extension' ); ?>
                    <?php self::render_sorting_icons(); ?>
                    </th>
                    <th class="indicator-accordion-sorting handle-accordion-sorting"><?php echo $total_plugin_db_upgrades . ' ' . _n( 'Update', 'Updates', $total_plugin_db_upgrades, 'mainwp-database-updater-extension' ); ?><?php self::render_sorting_icons(); ?></th>
                    <th class="no-sort right aligned">
                        <?php self::render_show_all_updates_button(); ?>
                    </th>
                </tr>
            </thead>
            <?php
            $site_view            = self::MAINWP_VIEW_PER_PLUGIN_THEME;
            $updates_table_helper = new MainWP_Database_Updater_Updates_Table_Helper( $site_view );
            $row_index            = 0;
            ?>
            <tbody id="plugins-updates-global" class="ui accordion">
                <?php foreach ( $allPlugins as $slug => $val ) : ?>
                    <?php
                    $cnt         = isset( $val['cnt'] ) ? intval( $val['cnt'] ) : 0;
                    $plugin_name = rawurlencode( $slug );
                    ?>
                    <tr class="ui title master-checkbox" plugin_slug="<?php echo $plugin_name; ?>">
                        <td class="accordion-trigger"><i class="icon dropdown"></i></td>
                        <td class="middle aligned">
                            <div class="ui master checkbox">
                                <input type="checkbox" name="">
                            </div>
                            <?php
                            $dir_slug = $pluginsInfo[ $slug ]['slug'];
                            $dir_slug = dirname( $dir_slug );
                            ?>
                            &nbsp;&nbsp;<?php echo apply_filters( 'mainwp_get_plugin_icon', '', $dir_slug ); ?>&nbsp;&nbsp;&nbsp;&nbsp;
                            <a href="<?php echo admin_url() . 'plugin-install.php?tab=plugin-information&plugin=' . esc_attr( $dir_slug ) . '&name=' . rawurlencode( $pluginsInfo[ $slug ]['name'] ); ?>" target="_blank" class="open-plugin-details-modal">
                                <?php echo esc_html( $pluginsInfo[ $slug ]['name'] ); ?>
                            </a>
                        </td>
                        <td sort-value="<?php echo $cnt; ?>"><?php echo $cnt; ?> <?php echo _n( 'Update', 'Updates', $cnt, 'mainwp-database-updater-extension' ); ?></td>
                        <td class="right aligned">
                            <?php if ( self::user_can_update_plugins() ) : ?>
                                <?php if ( 0 < $cnt ) : ?>
                                    <a href="javascript:void(0)" class="mainwp-update-selected-button ui mini basic button green " onClick="event.stopPropagation(); return mainwp_db_updater_updatesoverview_plugins_upgrade_all( '<?php echo $plugin_name; ?>', '<?php echo rawurlencode( $pluginsInfo[ $slug ]['name'] ); ?>', true )"><?php esc_html_e( 'Update Selected', 'mainwp-database-updater-extension' ); ?></a>
                                    <a href="javascript:void(0)" class="mainwp-update-all-button ui mini button green " onClick="event.stopPropagation();return mainwp_db_updater_updatesoverview_plugins_upgrade_all( '<?php echo $plugin_name; ?>', '<?php echo rawurlencode( $pluginsInfo[ $slug ]['name'] ); ?>' );"><?php esc_html_e( 'Update All', 'mainwp-database-updater-extension' ); ?></a>
                                <?php endif; ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php
                    ob_start();
                    ?>
                    <div id="mainwp-plugins-db-updates-childrow-holder-<?php echo esc_attr( $row_index ); ?>" style="display:none;">
                        <table id="mainwp-plugins-db-updates-childrow-<?php echo esc_attr( $row_index ); ?>" class="ui mainwp-manage-updates-table table plugins-bulk-updates not-default-init" style="width: 100%;" plugin_slug="<?php echo $plugin_name; ?>" plugin_name="<?php echo rawurlencode( $pluginsInfo[ $slug ]['name'] ); ?>">
                            <thead class="mainwp-768-hide">
                                <tr>
                                <?php $updates_table_helper->print_column_headers(); ?>
                                </tr>
                            </thead>
                            <tbody plugin_slug="<?php echo esc_attr( $plugin_name ); ?>">
                                <?php
                                MainWP_Database_Updater_Utility::data_seek( $websites );
                                while ( $websites && ( $website = MainWP_Database_Updater_Utility::fetch_object( $websites ) ) ) {
                                    if ( $website->is_ignorePluginUpdates ) {
                                        continue;
                                    }
                                    $plugin_db_upgrades = ! empty( $website->plugin_db_upgrades ) ? json_decode( $website->plugin_db_upgrades, true ) : array();
                                    if ( ! is_array( $plugin_db_upgrades ) ) {
                                        $plugin_db_upgrades = array();
                                    }

                                    $ignored_plugins = ! empty( $website->ignored_plugins ) ? json_decode( $website->ignored_plugins, true ) : array();
                                    if ( is_array( $ignored_plugins ) ) {
                                        $plugin_db_upgrades = array_diff_key( $plugin_db_upgrades, $ignored_plugins );
                                    }

                                    if ( ! isset( $plugin_db_upgrades[ $slug ] ) ) {
                                        continue;
                                    }
                                    $plugin_db_upgrade = $plugin_db_upgrades[ $slug ];

                                    $row_columns = array(
                                        'title'   => self::render_site_link_dashboard( $website, false ),
                                        'login'   => '<a href="admin.php?page=SiteOpen&newWindow=yes&websiteid=' . $website->id . '&_opennonce=' . wp_create_nonce( 'mainwp-admin-nonce' ) . '" class="open_newwindow_wpadmin" target="_blank"><i class="sign in icon"></i></a>',
                                        'version' => '<strong class="mainwp-768-show">' . esc_html__( 'Version: ', 'mainwp' ) . '</strong>' . esc_html( $plugin_db_upgrade['db_version'] ),
                                        'latest'  => '<strong class="mainwp-768-show">' . esc_html__( 'Latest: ', 'mainwp' ) . '</strong>' . esc_html( $plugin_db_upgrade['update']['new_db_version'] ),
                                        'client'  => ! empty( $website->client_name ) ? $website->client_name : '',
                                    );
                                    ?>
                                    <tr site_id="<?php echo esc_attr( $website->id ); ?>" site_name="<?php echo rawurlencode( stripslashes( $website->name ) ); ?>" updated="0">
                                        <?php
                                        $updates_table_helper->render_columns( $row_columns );
                                        ?>
                                        <td class="right aligned">
                                            <?php if ( self::user_can_update_plugins() ) : ?>
                                            <a href="javascript:void(0)" class="mainwp-update-now-button ui mini green button" onClick="return mainwp_db_updater_updatesoverview_plugins_upgrade( '<?php echo $plugin_name; ?>', <?php echo esc_attr( $website->id ); ?> )"><?php esc_html_e( 'Update Now', 'mainwp-database-updater-extension' ); ?></a>
                                        <?php endif; ?>
                                        </td>
                                    </tr>
                                    <?php
                                }
                                ?>
                            </tbody>
                        </table>
                    </div>
                    <?php
                    $child_rows[ $plugin_name ] = ob_get_clean();
                    ++$row_index;
                    ?>
                <?php endforeach; ?>
            </tbody>
            <tfoot>
                <tr>
                    <th class="collapsing no-sort"></th>
                    <th><?php esc_html_e( 'Plugin', 'mainwp-database-updater-extension' ); ?></th>
                    <th><?php echo $total_plugin_db_upgrades . ' ' . _n( 'Update', 'Updates', $total_plugin_db_upgrades, 'mainwp-database-updater-extension' ); ?></th>
                    <th class="no-sort right aligned"></th>
                </tr>
            </tfoot>
        </table>
        <?php
        foreach ( $child_rows as $child_tbl ) {
            echo $child_tbl; //phpcs:ignore --ok, hidden tables.
        }
    }

    /**
     * Method render_individual_plugins_updates()
     *
     * Render Plugins updates
     *
     * @param object $websites the websites.
     * @param int    $total_plugin_db_upgrades  Number of available db plugin updates.
     * @param mixed  $userExtension user extension.
     * @param array  $allPlugins all plugins.
     * @param array  $pluginsInfo pugins information.
     * @param array  $trustedPlugins trusted plugins.
     */
	public static function render_individual_plugins_updates( $websites, $total_plugin_db_upgrades, $userExtension, $allPlugins, $pluginsInfo, $trustedPlugins ) { // phpcs:ignore -- not quite complex method.
        ?>
        <?php if ( MainWP_Database_Updater_Utility::show_mainwp_message( 'mainwp-db-updater-info' ) ) : ?>
            <div class="ui info message">
                <i class="close icon mainwp-notice-dismiss" notice-id="mainwp-db-updater-info"></i>
                <?php printf( __( 'MainWP Database Updater Extension allows you to trigger detected Database updates for supported plugins. For more information, review %1$shelp documentation%2$s.', 'mainwp-database-updated-extension' ), '<a href="https://kb.mainwp.com/docs/mainwp-database-updater-extension/" target="_blank">', '</a>' ); ?>
            </div>
            <?php
        endif;

        MainWP_Database_Updater_Utility::data_seek( $websites );
        $website = MainWP_Database_Updater_Utility::fetch_object( $websites );

        if ( empty( $website ) ) {
            return;
        }

        $site_view            = self::MAINWP_VIEW_PER_PLUGIN_THEME;
        $updates_table_helper = new MainWP_Database_Updater_Updates_Table_Helper( $site_view );
        ?>
        <?php
        ob_start();
        ?>
            <table id="mainwp-db-plugins-updates-individual-table" class="ui mainwp-manage-updates-table table plugins-bulk-updates" site_id="<?php echo intval($website->id ); ?>" style="width: 100% !important;">
                <thead class="mainwp-768-hide">
                    <tr>
                    <?php $updates_table_helper->print_column_headers( true ); ?>
                </tr>
                </thead>
                <tbody>
                <?php
                foreach ( array( $website ) as $website ) :

                    if ( $website->is_ignorePluginUpdates ) {
                        continue;
                    }

                    $plugin_db_upgrades = ! empty( $website->plugin_db_upgrades ) ? json_decode( $website->plugin_db_upgrades, true ) : array();
                    if ( ! is_array( $plugin_db_upgrades ) ) {
                        $plugin_db_upgrades = array();
                    }

                    $ignored_plugins = ! empty( $website->ignored_plugins ) ? json_decode( $website->ignored_plugins, true ) : array();
                    if ( is_array( $ignored_plugins ) ) {
                        $plugin_db_upgrades = array_diff_key( $plugin_db_upgrades, $ignored_plugins );
                    }

                    ?>
                    <?php foreach ( $allPlugins as $slug => $val ) : ?>
                        <?php

                        if ( ! isset( $plugin_db_upgrades[ $slug ] ) ) {
                            continue;
                        }

                        $plugin_db_upgrade = $plugin_db_upgrades[ $slug ];

                        $row_columns = array(
                            'version' => '<strong class="mainwp-768-show">' . esc_html__( 'Version: ', 'mainwp' ) . '</strong>' . esc_html( $plugin_db_upgrade['db_version'] ),
                            'latest'  => '<strong class="mainwp-768-show">' . esc_html__( 'Latest: ', 'mainwp' ) . '</strong>' . esc_html( $plugin_db_upgrade['update']['new_db_version'] ),
                            'client'  => ! empty( $website->client_name ) ? $website->client_name : '',
                        );

                        $cnt         = isset( $val['cnt'] ) ? intval( $val['cnt'] ) : 0;
                        $plugin_name = rawurlencode( $slug );
                        ?>
                        <tr class="ui title" plugin_slug="<?php echo $plugin_name; ?>" site_id="<?php echo esc_attr( $website->id ); ?>" site_name="<?php echo rawurlencode( stripslashes( $website->name ) ); ?>" updated="0">
                            <td class="middle aligned">
                                <?php
                                $dir_slug = $pluginsInfo[ $slug ]['slug'];
                                $dir_slug = dirname( $dir_slug );
                                ?>
                                <?php echo apply_filters( 'mainwp_get_plugin_icon', '', $dir_slug ); ?>&nbsp;&nbsp;&nbsp;&nbsp;
                                <a href="<?php echo admin_url() . 'plugin-install.php?tab=plugin-information&plugin=' . esc_attr( $dir_slug ) . '&name=' . rawurlencode( $pluginsInfo[ $slug ]['name'] ); ?>" target="_blank" class="open-plugin-details-modal">
                                    <?php echo esc_html( $pluginsInfo[ $slug ]['name'] ); ?>
                                </a>
                            </td>
                            <?php
                            $updates_table_helper->render_columns( $row_columns, true );
                            ?>
                            <td class="right aligned">
                                <?php if ( self::user_can_update_plugins() ) : ?>
                                <a href="javascript:void(0)" class="mainwp-update-now-button ui mini green button" onClick="return mainwp_db_updater_updatesoverview_plugins_upgrade( '<?php echo $plugin_name; ?>', <?php echo esc_attr( $website->id ); ?> )"><?php esc_html_e( 'Update Now', 'mainwp-database-updater-extension' ); ?></a>
                            <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    <?php
                    endforeach;
                ?>
                </tbody>
            </table>
                    <?php
    }

                    /**
                     * Generates individual site overview page link.
                     *
                     * @param object $website The site object.
                     * @param bool   $echo Either echo or not.
                     *
                     * @return string Dashboard link.
                     */
    public static function render_site_link_dashboard( $website, $echo = true ) {
        $lnk = '<a href="' . admin_url( 'admin.php?page=managesites&dashboard=' . $website->id ) . '">' . stripslashes( $website->name ) . '</a>';
        if ( $echo ) {
            echo $lnk;
        } else {
            return $lnk;
        }
    }

                    /**
                     * Checks if the current user has permission to update plugins.
                     *
                     * @return bool Whether user can update plugins or not.
                     */
    public static function user_can_update_plugins() {
        if ( null === self::$user_can_update_plugins ) {
            if ( function_exists( '\mainwp_current_user_have_right' ) ) {
                self::$user_can_update_plugins = \mainwp_current_user_have_right( 'dashboard', 'update_plugins' ) ? true : false;
            } else {
                self::$user_can_update_plugins = true;
            }
        }
        return self::$user_can_update_plugins;
    }

                    /**
                     * Method render_sorting_icons()
                     *
                     * Render sorting up & down icons.
                     *
                     * @return void Render Sort up & down incon html.
                     */
    public static function render_sorting_icons() {
        ?>
        <i class="sort icon"></i><i class="sort up icon"></i><i class="sort down icon"></i>
        <?php
    }

                    /**
                     * Method render_show_all_updates_button()
                     *
                     * Render show all updates button.
                     *
                     * @return void Render Show All Updates button html.
                     */
    public static function render_show_all_updates_button() {
        ?>
        <a href="javascript:void(0)" class="ui mini button trigger-all-accordion">
            <?php
            /**
             * Filter: mainwp_show_all_updates_button_text
             *
             * Filters the Show All Updates button text.
             *
             * @since 4.1
             */
            echo esc_html__( 'Show All Updates', 'mainwp-database-updater-extension' );
            ?>
        </a>
        <?php
    }

                    /**
                     * Method get_user_extension()
                     *
                     * Get user extension.
                     *
                     * @return boolean|int false|get_user_extension_by_user_id()
                     *
                     * @uses \MainWP\Dashboard\MainWP_System::is_single_user()
                     */
    public function get_user_extension() {

        /**
         * Current user global.
         *
         * @global string
         */
        global $current_user;

        if ( empty( $current_user ) ) {
            $userid = 0;
        } else {
            $userid = $current_user->ID;
        }

        return $this->get_user_extension_by_user_id( $userid );
    }
}
