<?php
/**
 * =======================================
 * MainWP Database Updater Updates Table Helper
 * =======================================
 */
namespace MainWP\Extensions\DatabaseUpdater;

/**
 * Class MainWP_Updates_Table_Helper
 *
 * @package MainWP\Dashboard
 */
class MainWP_Database_Updater_Updates_Table_Helper {

    /**
     * Protected variable to hold columns info.
     *
     * @var array
     */
    protected $columns_info;

    /**
     * Protected variable to type
     *
     * @var string
     */
    public $type = 'plugin';

    /**
     * Protected variable to view per
     *
     * @var string
     */
    public $view_per;

    /**
     * MainWP_Updates_Table_Helper constructor.
     *
     * Run each time the class is called.
     *
     * @param string $view_per View per value.
     * @param string $type Type of plugin or theme, option, default: 'plugin'.
     */
    public function __construct( $view_per, $type = 'plugin' ) {
        $this->type     = $type;
        $this->view_per = $view_per;
    }

    /**
     * Method get_columns()
     *
     * Combine all columns.
     *
     * @param bool $individual true|false.
     *
     * @return array $columns Array of column names.
     */
    public function get_columns( $individual ) {

        if ( $individual ) {
            $title = esc_html__( 'Plugin', 'mainwp-database-updater-extension' );
        } else {
            $title = esc_html__( 'Website', 'mainwp-database-updater-extension' );
        }
        $columns = array(
            'title'   => $title,
            'login'   => '<i class="sign in alternate icon"></i>',
            'version' => esc_html__( 'DB Version', 'mainwp' ),
            'latest'  => esc_html__( 'Latest DB Version', 'mainwp' ),
            'client'  => esc_html__( 'Client', 'mainwp' ),
            'action'  => '',
        );
        if ( $individual ) {
            unset( $columns['login'] );
        }

        if ( ! \mainwp_current_user_can( 'dashboard', 'manage_clients' ) ) {
            unset( $columns['client'] );
        }

        return $columns;
    }

    /**
     * Get column info.
     *
     * @param bool $individual true|false.
     */
    protected function get_column_info( $individual ) {
        if ( isset( $this->columns_info ) ) {
            return $this->columns_info;
        }
        $columns            = $this->get_columns( $individual );
        $sortable           = $this->get_sortable_columns();
        $this->columns_info = array( $columns, $sortable );
        return $this->columns_info;
    }

    /**
     * Get sortable columns.
     *
     * @return array $sortable_columns Array of sortable column names.
     */
    public function get_sortable_columns() {
        $sortable_columns = array(
            'title'   => true,
            'version' => true,
        );
        return $sortable_columns;
    }

    /**
     * Echo the column headers.
     *
     * @param bool $individual true|false.
     */
    public function print_column_headers( $individual = false ) {

        list( $columns_header, $sortable ) = $this->get_column_info( $individual );

        foreach ( $columns_header as $column_key => $column_display_name ) {

            $class = array();
            if ( ! isset( $sortable[ $column_key ] ) ) {
                $class[] = 'no-sort';
            }

            if ( 'version' == $column_key || 'latest' == $column_key ) {
                $class[] = 'center aligned';
            }

            if ( ! empty( $class ) ) {
                $class = "class='" . join( ' ', $class ) . "'";
            } else {
                $class = '';
            }
            echo "<th $class>$column_display_name</th>";
        }
    }

    /**
     * Default column.
     *
     * @param mixed $value Value of column.
     * @param mixed $column_name Name of column.
     */
    public function column_default( $value, $column_name ) {
        $class = '';
        if ( 'version' == $column_name || 'latest' == $column_name ) {
            $class = 'mainwp-768-half-width-cell center aligned';
        }
        $col = '<td class="' . $class . '">';
        if ( 'title' == $column_name ) {
            $col .= '<div class="ui child checkbox">
			<input type="checkbox" name="">
		  </div>';
        }
        $col .= $value . '</td>';
        return $col;
    }

    /**
     *  Echo columns.
     *
     * @param array $columns Array of columns.
     * @param bool  $individual True|false.
     *
     * @return array Row columns.
     */
    public function render_columns( $columns, $individual = false ) {
        $row_columns            = $columns;
        list( $columns_header ) = $this->get_column_info( $individual );
        foreach ( $columns_header as $col => $title ) {
            if ( isset( $row_columns[ $col ] ) ) {
                $value = $row_columns[ $col ];
                if ( method_exists( $this, 'column_' . $col ) ) {
                    echo call_user_func( array( &$this, 'column_' . $col ), $value );
                } else {
                    echo $this->column_default( $value, $col );
                }
            }
        }
        return $row_columns;
    }
}
