<?php
/**
 * MainWP Jetpack Protect Overview.
 *
 * This file handles.
 *
 * @package MainWP/Extensions
 */

namespace MainWP\Extensions\JetpackProtect;

/**
 * Class MainWP_Jetpack_Protect_Overview
 */
class MainWP_Jetpack_Protect_Overview {

	/**
	 * Protected static variable to hold the single instance of the class.
	 *
	 * @var mixed Default null
	 */
	private static $instance = null;


	/**
	 * Enabled widgets
	 *
	 * @var array $enable_widgets
	 */
	private static $enable_widgets = array(
		'jetpack-protect'         => true,
		'jetpack-protect-details' => true,
	);

	/**
	 * Current page.
	 *
	 * @static
	 * @var string $page Current page.
	 */
	public static $page;

	/**
	 * Method get_instance()
	 *
	 * Create a new public static instance of
	 * MainWP_Jetpack_Protect_Overview().
	 *
	 * @return void $instance New public static Instance.
	 */
	static function get_instance() {
		if ( null == self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * MainWP_Jetpack_Protect_Overview class constructor.
	 *
	 * @return void
	 */
	public function __construct() {
		add_action( 'admin_init', array( &$this, 'admin_init' ) );
	}

	/**
	 * Initiates the admin page when loaded & sets initial Class Variables.
	 */
	public function admin_init() {
		// add_filter( 'screen_layout_columns', array( &$this, 'hook_screen_layout_columns' ), 10, 2 );
	}

	/**
	 * Method hook_managesites_subpage().
	 *
	 * @param array $subPage Input sub pages.
	 *
	 * @return array $subPage Output sub pages.
	 */
	public function hook_managesites_subpage( $subPage ) {
		$subPage[] = array(
			'title'            => __( 'Jetpack Protect', 'mainwp-jetpack-protect-extension' ),
			'slug'             => 'JetpackProtect',
			'sitetab'          => true,
			'menu_hidden'      => true,
			'callback'         => array( &$this, 'render_individual_page' ),
			'on_load_callback' => array( &$this, 'on_load_individual_overview_page' ),
		);
		return $subPage;
	}


	/**
	 * Render metabox
	 *
	 * Initiates the metabox.
	 */
	public static function render_metabox() {
		if ( isset( $_GET['page'] ) && 'managesites' == $_GET['page'] ) {
			$site_id = isset( $_GET['dashboard'] ) ? intval( $_GET['dashboard'] ) : 0; // individual site overview.
			self::render_individual_protect_data_widget( $site_id, true );
		} else {
			self::render_general_overview_widget();
		}
	}


	/**
	 * Method on_load_page()
	 *
	 * Run on page load.
	 */
	public function on_load_individual_overview_page() {
		$screen = get_current_screen();
		$page   = MainWP_Jetpack_Protect_Utility::get_page_id( $screen->id );

		$individual_overview = false;
		if ( isset( $_GET['page'] ) && 'ManageSitesJetpackProtect' == $_GET['page'] ) {
			$individual_overview = true;
		}

		if ( ! $individual_overview ) {
			return;
		}

		// $val = get_user_option( 'screen_layout_' . $page );
		// if ( ! $val ) {
		// global $current_user;
		// update_user_option( $current_user->ID, 'screen_layout_' . $page, 2, true );
		// }

		wp_enqueue_script( 'common' );
		wp_enqueue_script( 'wp-lists' );
		wp_enqueue_script( 'postbox' );
		wp_enqueue_script( 'dashboard' );
		wp_enqueue_script( 'widgets' );

		add_filter( 'mainwp_widget_boxes_number_columns', array( &$this, 'hook_widget_boxes_number_columns' ), 10, 2 );
		add_filter( 'mainwp_widget_boxes_show_widgets', array( &$this, 'hook_widget_boxes_show_widgets' ), 10, 2 );
		
		if ( isset( $_GET['tab'] ) && 'dashboard' == $_GET['tab'] ) {
		add_filter( 'mainwp_header_actions_right', array( &$this, 'hook_screen_options' ), 10, 2 );
		}

		$this->add_meta_boxes( $screen );
	}

	/**
	 * Method add_meta_boxes()
	 *
	 * Add MainWP Overview Page Widgets.
	 *
	 * @param array $page Current page.
	 */
	public function add_meta_boxes( $screen ) {

		$individualProtect_MetaBoxs = array();
		$individualProtect_MetaBoxs = apply_filters( 'mainwp_jetpack_protect_individual_overview_getmetaboxes', $individualProtect_MetaBoxs );
		foreach ( $individualProtect_MetaBoxs as $box ) {
			if ( isset( $box['plugin'] ) ) {
				$name                          = basename( $box['plugin'], '.php' );
				self::$enable_widgets[ $name ] = true;
			}
		}

		$values = self::$enable_widgets;

		$values               = apply_filters( 'mainwp_jetpack_protect_individual_overview_enabled_widgets', $values, null );
		self::$enable_widgets = array_merge( self::$enable_widgets, $values );

		// Load the Updates Overview widget.
		if ( self::$enable_widgets['jetpack-protect-details'] ) {
			do_action( 'mainwp_add_widget_box', 'jetpack-protect-details', array( &$this, 'hook_individual_protect_detail_widget' ), $screen, array( 1, 1, 7, 15 ) );
		}
		// Load the Updates Overview widget.
		if ( self::$enable_widgets['jetpack-protect'] ) {
			do_action( 'mainwp_add_widget_box', 'jetpack-protect', array( &$this, 'hook_individual_protect_widget' ), $screen, array( 1, 1, 5, 15 ) );
		}

		$i = 1;
		foreach ( $individualProtect_MetaBoxs as $metaBox ) {
			$enabled = true;
			if ( isset( $metaBox['plugin'] ) ) {
				$name = basename( $metaBox['plugin'], '.php' );
				if ( isset( self::$enable_widgets[ $name ] ) && ! self::$enable_widgets[ $name ] ) {
					$enabled = false;
				}
			}

			$id = isset( $metaBox['id'] ) ? $metaBox['id'] : $i++;
			$id = 'advanced-' . $id;

			if ( $enabled ) {
				do_action( 'mainwp_add_widget_box', $id, $metaBox['callback'], $screen, array( 1, 1, 3, 3 ) );
			}
		}
	}


	/**
	 * Method hook_screen_options()
	 *
	 * Create Page Settings button.
	 *
	 * @param mixed $input Screen options button HTML.
	 *
	 * @return mixed Screen sptions button.
	 */
	public function hook_screen_options( $input ) {
		return $input .
		'<a class="ui button basic icon" onclick="mainwp_jetpack_protect_overview_screen_options(); return false;" data-inverted="" data-position="bottom right" href="#" target="_blank" data-tooltip="' . esc_html__( 'Page Settings', 'mainwp-jetpack-protect-extension' ) . '">
			<i class="cog icon"></i>
		</a>';
	}


	/**
	 * Method hook_widget_boxes_number_columns()
	 *
	 * Create Page Settings button.
	 *
	 * @param int    $number number widget columns.
	 * @param string $page page.
	 *
	 * @return int number widget columns.
	 */
	public function hook_widget_boxes_number_columns( $number, $page ) {
		return get_user_option( 'mainwp_number_jetpack_protect_overview_columns' );
	}

	/**
	 * Method hook_widget_boxes_show_widgets()
	 *
	 * Get widgets.
	 *
	 * @param int    $show_widgets show widgets.
	 * @param string $page page.
	 *
	 * @return int number widget columns.
	 */
	public function hook_widget_boxes_show_widgets( $show_widgets, $page ) {
		return get_user_option( 'mainwp_jetpack_protect_show_widgets' );
	}

	/**
	 * Render individual page.
	 */
	public function render_individual_page() {
		do_action( 'mainwp_pageheader_sites', 'JetpackProtect' );
		$this->render_dashboard_body();
		do_action( 'mainwp_pagefooter_sites', 'JetpackProtect' );
	}

	/**
	 * Global Metabox
	 *
	 * Renders the Overview page widget content.
	 */
	public static function render_general_overview_widget() {
		?>
		<div class="ui grid mainwp-widget-header">
			<div class="twelve wide column">
				<h3 class="ui header handle-drag">
					<?php esc_html_e( 'Jetpack Protect', 'mainwp-jetpack-protect-extension' ); ?>
					<div class="sub header"><?php esc_html_e( 'Vulnerable sites detected by the Jetpack Protect plugin.', 'mainwp-jetpack-protect-extension' ); ?></div>
				</h3>
			</div>
		</div>
		<div class="mainwp-scrolly-overflow">
		<?php MainWP_Jetpack_Protect_Plugin::get_instance()->gen_dashboard_widget_table(); ?>
		</div>
		
		<div class="ui two columns grid mainwp-widget-footer">
			<div class="left aligned column">
				<a href="admin.php?page=Extensions-Mainwp-Jetpack-Protect-Extension" class="ui basic green mini fluid button"><?php esc_html_e( 'Jetpack Protect Dashboard', 'mainwp-jetpack-protect-extension' ); ?></a>
			</div>
		</div>
		<?php
	}

	/**
	 * Individual Metabox
	 *
	 * Renders the individual site Overview page widget content.
	 */
	public function hook_individual_protect_widget() {
		$id = isset( $_GET['id'] ) ? intval( $_GET['id'] ) : 0; // individual site overview.
		self::render_individual_protect_data_widget( $id );
	}

	/**
	 * Individual detail protect Metabox
	 *
	 * Renders the individual site detail protect page widget content.
	 */
	public function hook_individual_protect_detail_widget() {
		$id = isset( $_GET['id'] ) ? intval( $_GET['id'] ) : 0; // individual site overview.
		$this->render_individual_detail_data_widget( $id );
	}


	/**
	 * Individual protect Metabox.
	 *
	 * Renders the individual site Overview page widget content.
	 *
	 * @param mixed $site_id Individual site ID.
	 */
	public static function render_individual_protect_data_widget( $site_id, $is_indi_overview = false ) {

		if ( empty( $site_id ) ) {
			return;
		}

		$status_data   = array();
		$lasttime_scan = 0;
		$num_vuln      = false;

		$plugins = array();
		$themes  = array();
		$files    = array();
		$database = array();

		$sql_sites  = MainWP_Jetpack_Protect_DB::get_instance()->get_sql_websites_ext_data( $site_id );
		$dbwebsites = MainWP_Jetpack_Protect_DB::get_instance()->query( $sql_sites );

		$protect = false;
		if ( $dbwebsites ) {
			$protect = MainWP_Jetpack_Protect_DB::fetch_object( $dbwebsites );
		}

		if ( ! empty( $protect ) ) {
			$lasttime_scan       = intval( $protect->lasttime_scan );
			$last_protect_status = MainWP_Jetpack_Protect_DB::get_instance()->get_protect_status_by( 'wpid', $site_id, 1 );
			if ( ! empty( $last_protect_status ) ) {
				$status_data = ! empty( $last_protect_status->status_data ) ? json_decode( $last_protect_status->status_data, true ) : array();
				if ( ! is_array( $status_data ) ) {
					$status_data = array();
				}
				if ( isset( $status_data['num_threats'] ) ) {
					$num_vuln = intval( $status_data['num_threats'] );
				}
			}

			if ( isset( $protect->plugins ) && ! empty( $protect->plugins ) ) {
				$plugins = json_decode( $protect->plugins, true );
			}
			if ( ! is_array( $plugins ) ) {
				$plugins = array();
			}
			if ( isset( $protect->themes ) && ! empty( $protect->themes ) ) {
				$themes = json_decode( $protect->themes, true );
			}
			if ( ! is_array( $themes ) ) {
				$themes = array();
			}
			if ( isset( $protect->files ) && ! empty( $protect->files ) ) {
				$files = json_decode( $protect->files, true );
			}
			if ( ! is_array( $files ) ) {
				$files = array();
			}
			if ( isset( $protect->database ) && ! empty( $protect->database ) ) {
				$database = json_decode( $protect->database, true );
			}
			if ( ! is_array( $database ) ) {
				$database = array();
			}
		}

		if ( ! empty( $status_data ) ) {

			$num_core_vuln    = isset( $status_data['core'] ) && isset( $status_data['core']['threats'] ) && is_array( $status_data['core']['threats'] ) ? count( $status_data['core']['threats'] ) : false;
			$num_plugins_vuln = isset( $status_data['num_plugins_threats'] ) ? intval( $status_data['num_plugins_threats'] ) : false;
			$num_themes_vuln  = isset( $status_data['num_themes_threats'] ) ? intval( $status_data['num_themes_threats'] ) : false;
			$num_files_vuln    = isset( $status_data['files'] ) ? count( $status_data['files'] ) : false;
			$num_database_vuln = isset( $status_data['database'] ) ? count( $status_data['database'] ) : false;

			$lists_details = array(
				'core'    => array(
					'title'    => __( 'WordPress', 'mainwp-jetpack-protect-extension' ),
					'num_vuln' => $num_core_vuln,
					'items'    => isset( $status_data['core'] ) ? $status_data['core'] : array(),
				),
				'plugins' => array(
					'title'    => __( 'Plugins', 'mainwp-jetpack-protect-extension' ),
					'num_vuln' => $num_plugins_vuln,
					'items'    => isset( $status_data['plugins'] ) ? $status_data['plugins'] : array(),
				),
				'themes'  => array(
					'title'    => __( 'Themes', 'mainwp-jetpack-protect-extension' ),
					'num_vuln' => $num_themes_vuln,
					'items'    => isset( $status_data['themes'] ) ? $status_data['themes'] : array(),
				),
				'files'    => array(
					'title'    => __( 'Files', 'mainwp-jetpack-protect-extension' ),
					'num_vuln' => $num_files_vuln,
					'items'    => isset( $status_data['files'] ) ? $status_data['files'] : array(),
				),
				'database' => array(
					'title'    => __( 'Database', 'mainwp-jetpack-protect-extension' ),
					'num_vuln' => $num_database_vuln,
					'items'    => isset( $status_data['database'] ) ? $status_data['database'] : array(),
				),

			);
		}

		?>
		<div class="ui grid mainwp-widget-header">
			<div class="twelve wide column">
				<h3 class="ui header handle-drag">
					<?php echo ( false !== $num_vuln ) ? intval( $num_vuln ) . ' ' . esc_html__( 'vulnerabilities found.', 'mainwp-jetpack-protect-extension' ) : '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'; ?>
					<div class="sub header"><?php echo ( false !== $num_vuln ) ? esc_html__( 'Latest results as of', 'mainwp-jetpack-protect-extension' ) . ' ' . ( $lasttime_scan ? MainWP_Jetpack_Protect_Utility::format_timestamp( $lasttime_scan ) : '' ) : '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;'; ?></div>
				</h3>
			</div>
		</div>
		
		<?php

		if ( empty( $protect ) ) {
			?>
			<div class="mainwp-scrolly-overflow">
			<h2 class="ui icon header">
				<i class="info circle icon"></i>
				<div class="content">
					<?php esc_html_e( 'Jetpack Protect Not Detected', 'mainwp-jetpack-protect-extension' ); ?>
					<div class="sub header"><?php esc_html_e( 'First install and activate the Jetpack Protect plugin on the child site.', 'mainwp-jetpack-protect-extension' ); ?></div>
					<div class="ui hidden divider"></div>
				</div>
			</h2>
			</div>
			<?php
			return;
		}

		if ( ! empty( $status_data ) ) {
			?>
			<div class="mainwp-scrolly-overflow">
			<table class="ui table" id="mainwp-jetpack-protect-vulnerabilities-table" website-id="<?php echo intval( $site_id ); ?>" status-id="<?php echo $last_protect_status ? intval( $last_protect_status->statusid ) : 0; ?>">
				<tbody class="ui accordion">
					<tr>
						<td></td>
						<td><?php esc_html_e( 'All Vulnerabilities', 'mainwp-jetpack-protect-extension' ); ?></td>
						<td class="right aligned">
							<?php if ( isset( $num_vuln ) && $num_vuln > 0 ) : ?>
							<span data-tooltip="<?php esc_attr_e( 'Number of vulnerabilities.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted="">
								<a href="javascript:void(0)" class="ui big circular red label mainwp-all-jp-vulnerabilities-label" style="color:#fff!important;">
									<?php echo intval( $num_vuln ); ?>
								</a>
							</span>
							<?php else : ?>
								<i class="green check icon"></i>
							<?php endif; ?>
						</td>
					</tr>
				<?php foreach ( $lists_details as $type => $detail ) : ?>
						<?php
						$items = $detail['items'];
						if ( ! is_array( $items ) ) {
							$items = array();
						}
						$num_vuln = intval( $detail['num_vuln'] );
						$v_color  = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_vuln );

						?>
						<?php if ( 'core' === $type || 'plugins' === $type || 'themes' === $type ) : ?>
						<tr class="ui title">
							<td  class="accordion-trigger collapsing"><i class="icon dropdown"></i></td>
							<td><span data-inverted="" data-tooltip="<?php echo esc_attr( __( 'Click to see details', 'mainwp-jetpack-protect-extension' ) ); ?>"><?php echo esc_html( $detail['title'] ); ?></span></td>
							<td class="right aligned">
							<?php if ( false === $detail['num_vuln'] ) : ?>
								<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
							<?php elseif ( 0 == $detail['num_vuln'] ) : ?>
								<i class="green large check icon"></i>
							<?php else : ?>
								<span data-tooltip="<?php esc_attr_e( 'Number of vulnerabilities.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted="">
									<a href="javascript:void(0)" class="ui large circular label <?php echo $v_color; ?> mainwp-<?php echo $type; ?>-jp-vulnerabilities-label" style="color:#fff!important;"><?php echo intval( $detail['num_vuln'] ); ?></a>
								</span>
							<?php endif; ?>
							</td>
						</tr>
						<?php elseif ( 'files' === $type && 0 < $detail['num_vuln'] ) : ?>
						<tr>
							<td></td>
							<td><span><?php echo esc_html( $detail['title'] ); ?></span></td>
							<td class="right aligned">
							<?php if ( false === $detail['num_vuln'] ) : ?>
								<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
							<?php elseif ( 0 == $detail['num_vuln'] ) : ?>
								<i class="green large check icon"></i>
							<?php else : ?>
								<span data-tooltip="<?php esc_attr_e( 'Number of vulnerabilities.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted="">
									<a href="javascript:void(0)" class="ui large circular label <?php echo $v_color; ?> mainwp-<?php echo $type; ?>-jp-vulnerabilities-label" style="color:#fff!important;"><?php echo intval( $detail['num_vuln'] ); ?></a>
								</span>
							<?php endif; ?>
							</td>
						</tr>
						<?php elseif ( 'database' === $type && 0 < $detail['num_vuln'] ) : ?>
						<tr>
							<td></td>
							<td><span><?php echo esc_html( $detail['title'] ); ?></span></td>
							<td class="right aligned">
							<?php if ( false === $detail['num_vuln'] ) : ?>
								<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
							<?php elseif ( 0 == $detail['num_vuln'] ) : ?>
								<i class="green large check icon"></i>
							<?php else : ?>
								<span data-tooltip="<?php esc_attr_e( 'Number of vulnerabilities.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted="">
									<a href="javascript:void(0)" class="ui large circular label <?php echo $v_color; ?> mainwp-<?php echo $type; ?>-jp-vulnerabilities-label" style="color:#fff!important;"><?php echo intval( $detail['num_vuln'] ); ?></a>
								</span>
							<?php endif; ?>
							</td>
						</tr>
						<?php endif; ?>
						<?php
						$found = is_array( $items ) ? count( $items ) : false;
						?>
						<tr class="content">
							<td colspan="3">
							<?php if ( $found ) : ?>
								<table id="mainwp-jetpack-protect-items-inner-table" class="ui table mainwp-manage-updates-table mainwp-updates-list">
									<tbody>	
									<?php
									if ( 'core' === $type ) {
										$name     = isset( $items['name'] ) ? esc_html( $items['name'] ) : '';
										$num_vuln = isset( $items['threats'] ) && is_array( $items['threats'] ) ? count( $items['threats'] ) : false;
										$version  = isset( $items['version'] ) ? esc_html( $items['version'] ) : '';
										$v_color  = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_vuln );
										$icon     = '<i class="wordpress icon"></i>'; // phpcs:ignore -- to fix format auto 'wordpress' word.
										?>
										<tr class="mainwp-jetpack-protect-detail-item" item-type="core" item-slug="core">
											<td class="collapsing">
												<?php echo $icon; ?>&nbsp;&nbsp;&nbsp;&nbsp;
												<?php
													echo esc_html( $name . ' ' . $version );
												?>
											</td>
											<td>
												<?php if ( false === $num_vuln ) : ?>
													<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
												<?php elseif ( 0 == $num_vuln ) : ?>
													<i class="green large check icon"></i>
												<?php else : ?>
													<span data-tooltip="<?php esc_attr_e( 'Number of vulnerabilities.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted="">
														<a href="javascript:void(0)" class="ui large circular label <?php echo $v_color; ?>" style="color:#fff!important;">
															<?php echo intval( $num_vuln ); ?>
														</a>
													</span>
												<?php endif; ?>
											</td>
										</tr>
										<?php
									} else {
										?>
										<?php foreach ( $items as $item ) : ?>
											<?php
											$slug     = isset( $item['slug'] ) ? esc_html( $item['slug'] ) : '';
											$name     = isset( $item['name'] ) ? esc_html( $item['name'] ) : '';
											$num_vuln = isset( $item['threats'] ) && is_array( $item['threats'] ) ? count( $item['threats'] ) : false;
											$v_color  = MainWP_Jetpack_Protect_Utility::get_vulnerabilities_number_color( $num_vuln );

											$dir_slug = '';
											$icon     = '';
											if ( ! empty( $slug ) ) {
												$dir_slug = apply_filters( 'mainwp_get_dir_slug', $slug, $slug );
												if ( 'plugins' === $type ) {
													$icon = apply_filters( 'mainwp_get_plugin_icon', '', $dir_slug );
												} elseif ( 'themes' === $type ) {
													$icon = apply_filters( 'mainwp_get_theme_icon', '', $dir_slug );
												}
											}

											$link_detail = '';
											if ( 'plugins' === $type ) {
												$link_detail = admin_url() . 'plugin-install.php?tab=plugin-information&plugin=' . rawurlencode( $dir_slug );
											}
											?>
											<tr class="mainwp-jetpack-protect-detail-item" item-type="<?php echo esc_attr( $type ); ?>" item-slug="<?php echo esc_attr( $slug ); ?>">
												<td class="collapsing">
													<?php echo $icon; ?>&nbsp;&nbsp;&nbsp;&nbsp;
													<?php
													if ( ! empty( $link_detail ) ) {
														?>
														<a href="<?php echo esc_url( $link_detail ); ?>" target="_blank" class="open-plugin-details-modal"><?php echo esc_html( $name ); ?></a>
														<?php
													} else {
														echo esc_html( $name );
													}
													?>
												</td>
												<td>
													<?php if ( false === $num_vuln ) : ?>
														<span data-tooltip="<?php esc_attr_e( 'No data available for the site.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted=""><i class="exclamation circle icon"></i></span>
													<?php elseif ( 0 == $num_vuln ) : ?>
														<i class="green large check icon"></i>
													<?php else : ?>
														<span data-tooltip="<?php esc_attr_e( 'Number of vulnerabilities.', 'mainwp-jetpack-protect-extension' ); ?>" data-position="left center" data-inverted="">
															<a href="javascript:void(0)" item-slug="<?php echo esc_attr( $slug ); ?>" class="ui large circular label <?php echo $v_color; ?> mainwp-item-jp-vulnerabilities-label" style="color:#fff!important;"><?php echo intval( $num_vuln ); ?></a>
														</span>
													<?php endif; ?>
												</td>
											</tr>
									<?php endforeach; ?>
									<?php }; ?>
									</tbody>
								</table>
								<?php endif; ?>
							</td>
						</tr>
					<?php endforeach; ?>
				</tbody>
			</table>
		</div>
		<?php }; ?>

		<div class="ui two columns grid mainwp-widget-footer">
			<div class="column">
				<a href="admin.php?page=Extensions-Mainwp-Jetpack-Protect-Extension" class="ui basic green mini fluid button"><?php esc_html_e( 'Jetpack Protect Overview', 'mainwp-jetpack-protect-extension' ); ?></a>
			</div>
		<?php if ( $is_indi_overview ) { ?>	
			<div class="column">
				<a href="admin.php?page=ManageSitesJetpackProtect&id=<?php echo $site_id; ?>" class="ui green mini fluid button"><?php esc_html_e( 'Vulnerabilities Details', 'mainwp-jetpack-protect-extension' ); ?></a>
			</div>
			<?php } ?>
		</div>

		<?php

		$table_features = array(
			'searching' => 'false',
			'paging'    => 'false',
			'stateSave' => 'true',
			'info'      => 'false',
			'exclusive' => 'false',
			'duration'  => '200',
		);
		?>
		<script type="text/javascript">
			jQuery( document ).ready( function () {
				jQuery( '#mainwp-jetpack-protect-vulnerabilities-table .ui.accordion' ).accordion( {
					"exclusive": <?php echo $table_features['exclusive']; ?>,
					"duration": <?php echo $table_features['duration']; ?>,
					onOpen: function(){
					}
				} );
			} );
		</script>
		<?php
	}

	/**
	 * Individual detail Metabox
	 *
	 * Renders the individual site Overview page widget content.
	 */
	public function render_individual_detail_data_widget() {
		$site_id = isset( $_GET['id'] ) ? intval( $_GET['id'] ) : 0;

		if ( empty( $site_id ) ) {
			return;
		}

		$status_data   = array();
		$lasttime_scan = 0;
		$all_num_vuln  = 0;

		$protect = MainWP_Jetpack_Protect_DB::get_instance()->get_protect_by( 'wpid', $site_id );
		if ( ! empty( $protect ) ) {
			$lasttime_scan = intval( $protect->lasttime_scan );

			$last_protect_status = MainWP_Jetpack_Protect_DB::get_instance()->get_protect_status_by( 'wpid', $site_id, 1 );
			if ( ! empty( $last_protect_status ) ) {
				$status_data = ! empty( $last_protect_status->status_data ) ? json_decode( $last_protect_status->status_data, true ) : array();
				if ( ! is_array( $status_data ) ) {
					$status_data = array();
				}
			}
		}

		if ( ! empty( $status_data ) ) {
			$num_core_vuln    = isset( $status_data['core'] ) && isset( $status_data['core']['threats'] ) && is_array( $status_data['core']['threats'] ) ? count( $status_data['core']['threats'] ) : false;
			$num_plugins_vuln = isset( $status_data['num_plugins_threats'] ) ? intval( $status_data['num_plugins_threats'] ) : false;
			$num_themes_vuln  = isset( $status_data['num_themes_threats'] ) ? intval( $status_data['num_themes_threats'] ) : false;
			$num_files_vuln    = isset( $status_data['files'] ) ? count( $status_data['files'] ) : false;
			$num_database_vuln = isset( $status_data['database'] ) ? count( $status_data['database'] ) : false;

			$lists_details = array(
				'core'    => array(
					'title'    => __( 'WordPress', 'mainwp-jetpack-protect-extension' ),
					'num_vuln' => $num_core_vuln,
					'items'    => isset( $status_data['core'] ) ? $status_data['core'] : array(),
				),
				'plugins' => array(
					'title'    => __( 'Plugins', 'mainwp-jetpack-protect-extension' ),
					'num_vuln' => $num_plugins_vuln,
					'items'    => isset( $status_data['plugins'] ) ? $status_data['plugins'] : array(),
				),
				'themes'  => array(
					'title'    => __( 'Themes', 'mainwp-jetpack-protect-extension' ),
					'num_vuln' => $num_themes_vuln,
					'items'    => isset( $status_data['themes'] ) ? $status_data['themes'] : array(),
				),
				'files'    => array(
					'title'    => __( 'Files', 'mainwp-jetpack-protect-extension' ),
					'num_vuln' => $num_files_vuln,
					'items'    => isset( $status_data['files'] ) ? $status_data['files'] : array(),
				),
				'database' => array(
					'title'    => __( 'Database', 'mainwp-jetpack-protect-extension' ),
					'num_vuln' => $num_database_vuln,
					'items'    => isset( $status_data['database'] ) ? $status_data['database'] : array(),
				),

			);
		}

		?>
		<div class="ui grid mainwp-widget-header">
			<div class="twelve wide column">
				<h3 class="ui header handle-drag">
					<?php esc_html_e( 'All Vulnerabilities', 'mainwp-jetpack-protect-extension' ); ?>
					<div class="sub header"><?php esc_html_e( 'Fix detected vulnerabilities by Updating or Deleting affected items.', 'mainwp-jetpack-protect-extension' ); ?></div>
				</h3>
			</div>
		</div>
		<?php

		if ( empty( $protect ) ) {
			?>
			<div class="mainwp-scrolly-overflow">
			<h2 class="ui icon header">
				<i class="info circle icon"></i>
				<div class="content">
					<?php esc_html_e( 'Jetpack Protect Not Detected', 'mainwp-jetpack-protect-extension' ); ?>
					<div class="sub header"><?php esc_html_e( 'First install and activate the Jetpack Protect plugin on the child site.', 'mainwp-jetpack-protect-extension' ); ?></div>
					<div class="ui hidden divider"></div>
				</div>
			</h2>
		</div>
			<?php
			return;
		}

		if ( ! empty( $status_data ) ) {
			?>
			<div class="mainwp-scrolly-overflow">
			<div class="mainwp-all-vul-widget">
			<table class="ui tablet stackable table" id="mainwp-jetpack-protect-vulnerabilities-table" website-id="<?php echo intval( $site_id ); ?>" status-id="<?php echo $last_protect_status ? intval( $last_protect_status->statusid ) : 0; ?>">
				<tbody class="ui accordion">
				<?php foreach ( $lists_details as $type => $type_details ) : ?>
						<?php
						$items = $type_details['items'];
						if ( ! is_array( $items ) ) {
							$items = array();
						}
						if ( empty( $items ) ) {
							continue;
						}

						if ( 'core' === $type ) {
							$vuln_items = isset( $items['threats'] ) && is_array( $items['threats'] ) ? $items['threats'] : array();
							if ( empty( $vuln_items ) ) {
								continue;
							}
							$all_num_vuln += count( $vuln_items );
							$this->render_row_detail_data_widget( $vuln_items, $type, $items, $site_id );
						} else if ( 'plugins' === $type || 'themes' === $type ) {
							foreach ( $items as $detail ) :
								$vuln_items = isset( $detail['threats'] ) && is_array( $detail['threats'] ) ? $detail['threats'] : array();
								if ( empty( $vuln_items ) ) {
									continue;
								}
								$all_num_vuln += count( $vuln_items );
								$this->render_row_detail_data_widget( $vuln_items, $type, $detail, $site_id );
								?>
								<?php
						endforeach; // foreach $items.
						} else if ( 'files' === $type ) {
							$vuln_items = isset( $items ) && is_array( $items ) ? $items : array();
							if ( empty( $vuln_items ) ) {
								continue;
							}
							$all_num_vuln += count( $vuln_items );
							$this->render_row_detail_data_widget( $vuln_items, $type, $items, $site_id );
						}
						?>
				<?php endforeach; // foreach $lists_details. ?>
				</tbody>
			</table>
			</div>
		</div>
		<?php }; ?>
		
		<?php

		$table_features = array(
			'searching' => 'false',
			'paging'    => 'false',
			'stateSave' => 'true',
			'info'      => 'false',
			'exclusive' => 'false',
			'duration'  => '200',
		);
		?>
		<script type="text/javascript">
			jQuery( document ).ready( function () {
				jQuery( '#mainwp-jetpack-protect-vulnerabilities-table .ui.accordion' ).accordion( {
					"exclusive": <?php echo $table_features['exclusive']; ?>,
					"duration": <?php echo $table_features['duration']; ?>,
					onOpen: function(){
					}
				} );
			} );
		</script>
		<?php
		do_action( 'mainwp_render_plugin_details_modal' );
	}


	/**
	 * Individual detail Metabox
	 *
	 * Renders the individual site Overview page widget content.
	 *
	 * @param array  $vuln_items vulnerabilities items.
	 * @param string $type vulnerabilities type.
	 * @param array  $detail protect status detail.
	 * @param int    $site_id The website ID.
	 *
	 * @return void
	 */
	public function render_row_detail_data_widget( $vuln_items, $type, $detail, $site_id ) {

		$name        = isset( $detail['name'] ) ? esc_html( $detail['name'] ) : '';
		$slug        = isset( $detail['slug'] ) ? esc_html( $detail['slug'] ) : '';
		$version     = isset( $detail['version'] ) ? esc_html( $detail['version'] ) : '';
		$link_detail = '';

		if ( 'plugins' === $type && ! empty( $slug ) ) {
			$dir_slug    = apply_filters( 'mainwp_get_dir_slug', $slug, $slug );
			$link_detail = admin_url() . 'plugin-install.php?tab=plugin-information&plugin=' . rawurlencode( $dir_slug );
		}

		if ( 'plugins' === $type ) {
			$icon = '<span data-inverted="" data-position="left center" data-tooltip="' . esc_attr__( 'Plugin threat', 'mainwp-jetpack-protect-extension' ) . '"><i class="plug large icon"></i></span>';
		} elseif ( 'themes' === $type ) {
			$icon = '<span data-inverted="" data-position="left center" data-tooltip="' . esc_attr__( 'Theme threat', 'mainwp-jetpack-protect-extension' ) . '"><i class="tint large icon"></i></span>';
		} elseif ( 'files' === $type ) {
			$icon = '<span data-inverted="" data-position="left center" data-tooltip="' . esc_attr__( 'File threat', 'mainwp-jetpack-protect-extension' ) . '"><i class="code large icon"></i></span>';
		} elseif ( 'database' === $type ) {
			$icon = '<span data-inverted="" data-position="left center" data-tooltip="' . esc_attr__( 'Database threat', 'mainwp-jetpack-protect-extension' ) . '"><i class="border all large icon"></i></span>';
		} else {
			$icon = '<span data-inverted="" data-position="left center" data-tooltip="' . esc_attr__( 'WP Core threat', 'mainwp-jetpack-protect-extension' ) . '"><i class="wordpress large icon"></i></span>'; //phpcs:ignore -- to fix WordPress word. 
		}

		foreach ( $vuln_items as $vuln_item ) :
			$vuln_id     = isset( $vuln_item['id'] ) ? $vuln_item['id'] : '';
			$title       = isset( $vuln_item['title'] ) ? $vuln_item['title'] : '';
			$fixed_in    = isset( $vuln_item['fixed_in'] ) ? $vuln_item['fixed_in'] : '';
			$description = isset( $vuln_item['description'] ) ? $vuln_item['description'] : '';
			$signature   = isset( $vuln_item['signature'] ) ? $vuln_item['signature'] : '';
			$filename    = isset( $vuln_item['filename'] ) ? $vuln_item['filename'] : '';
			$source      = isset( $vuln_item['source'] ) ? $vuln_item['source'] : '';
			?>
				<tr class="ui title" item-slug="<?php echo esc_attr( $slug ); ?>" type="<?php echo esc_attr( $type ); ?>" item-type="<?php echo esc_attr( $type ); ?>">
					<td  class="accordion-trigger collapsing"><i class="icon dropdown"></i></td>
					<td>
						<h4 class="ui header">
						<?php if ( $version == $fixed_in && 'files' !== $type ) : ?>
								<i class="cog loading icon"></i>
								<?php else : ?>
							<?php echo $icon; ?>
								<?php endif; ?>
							<div class="content">
						<?php
						if ( 'core' === $type ) {
									echo esc_html( $name . ' ' . $version );
								} else  if ( 'plugins' === $type || 'themes' === $type ) {
							if ( 'plugins' === $type && ! empty( $link_detail ) ) {
								?>
										<a href="<?php echo esc_url( $link_detail ); ?>" target="_blank" class="open-plugin-details-modal"><?php echo esc_html( $name . ' ' . $version ); ?></a>
								<?php
							} else {
										echo esc_html( $name . ' ' . $version );
							}
								} else if ( 'files' === $type ) {
									echo esc_html( $title );
						}
						?>
								<div class="sub header">
									<?php if ( 'files' === $type ) : ?>
										<?php echo esc_html( $signature ); ?>
									<?php else : ?>
										<?php echo esc_html( $title ); ?>
									<?php endif; ?>
								</div>
							</div>
						</h4>
					</td>
				</tr>
				<tr class="content" item-slug="<?php echo esc_attr( $slug ); ?>" item-type="<?php echo esc_attr( $type ); ?>" website-id="<?php echo intval( $site_id ); ?>">
					<td></td>
					<td colspan="2">
					<div class="ui segment">
							<p><strong><?php esc_html_e( 'What is the problem?', 'mainwp-jetpack-protect-extension' ); ?></strong></p>
							<p><?php echo esc_html( $description ); ?></p>
							<?php if ( 'files' !== $type ) : ?>
							<p><?php echo sprintf( esc_html__( '%1$sSee more technical details of this vulnerability%2$s', 'mainwp-jetpack-protect-extension' ), '<a href="' . esc_attr( $source ) . '" target="_blank">', '</a>' ); ?></p>
							<?php endif; ?>
							<?php if ( $version == $fixed_in && 'files' !== $type ) : ?>
							<div class="ui info message"><?php esc_html_e( 'Fix submitted. Changes will be reflected after the Jetpack Protect completes rescan process. Please try to resync your MainWP Dashboard later.', 'mainwp-jetpack-protect-extension' ); ?></div>	
							<?php else : ?>
							<?php if ( 'files' !== $type ) : ?>
							<p><strong><?php esc_html_e( 'How to fix it?', 'mainwp-jetpack-protect-extension' ); ?></strong></p>
							<?php endif; ?>
							<?php if ( 'files' === $type ) : ?>
							<p><strong><?php esc_html_e( 'File:', 'mainwp-jetpack-protect-extension' ); ?></strong></p>
							<p><?php echo esc_html( $filename ); ?></p>
							<?php endif; ?>
							
							<p>
							<?php if ( ! empty( $fixed_in ) ) { ?>
								<a href="javascript:void(0)" class="ui basic mini green button mainwp-jetpack-protect-widget-update-item"><?php echo esc_html__( 'Update to', 'mainwp-jetpack-protect-extension' ) . ' ' . esc_html( $name ) . ' ' . esc_html( $fixed_in ); ?></a>
							<?php } ?>
							<?php if ( 'core' !== $type && 'files' !== $type && 'database' !== $type ) { ?>
								<a href="javascript:void(0)" class="ui mini green button mainwp-jetpack-protect-widget-delete-item"><?php echo esc_html__( 'Delete', 'mainwp-jetpack-protect-extension' ) . ' ' . esc_html( $name ) . ' ' . esc_html( $version ); ?></a>
							<?php } ?>
							</p>
							<?php endif; ?>
							
							<div class="ui active inverted dimmer mainwp-loader-element" style="display:none">
							<div class="ui text loader"><?php esc_html_e( 'Please wait...', 'mainwp-jetpack-protect-extension' ); ?></div>
					  </div>
					</div>
					</td>
				</tr>
			<?php
			endforeach; // foreach $vuln_items
	}

	/**
	 * Method render_dashboard_body()
	 *
	 * Render the Dashboard Body content.
	 *
	 * @param mixed $dashboard     Dashboard.
	 */
	public function render_dashboard_body() {
		$screen = get_current_screen();
		?>
		<div class="mainwp-primary-content-wrap">
			
		<div id="mainwp-message-zone" class="ui message" style="display:none;"></div>

			<?php if ( MainWP_Jetpack_Protect_Utility::show_mainwp_message( 'protect-widgets' ) ) : ?>
				<div class="ui info message">
				<?php echo sprintf( esc_html__( 'To hide or show a widget, click the Cog (%s) icon and select options from "Show widgets"', 'mainwp-jetpack-protect-extension' ), '<i class="cog icon"></i>' ); ?>
					<i class="close icon mainwp-notice-dismiss" notice-id="protect-widgets"></i>
			</div>
		<?php endif; ?>
		<?php if ( MainWP_Jetpack_Protect_Utility::show_mainwp_message( 'protect-info-notice' ) ) : ?>
			<div class="ui info message">
				<?php echo sprintf( esc_html__( 'Jetpack Protect automatically scans your child sites once per day. This means that after resolving vulnerability issues, it can take up to 24 hours before the extension reflects changes.', 'mainwp-jetpack-protect-extension' ), '<i class="cog icon"></i>' ); ?>
				<i class="close icon mainwp-notice-dismiss" notice-id="protect-info-notice"></i>
				</div>
			<?php endif; ?>

			<?php
			/**
			 * Action: mainwp_before_jetpack_protect_overview_widgets
			 *
			 * Fires at the top of the Overview page (before first widget).
			 *
			 * @since 4.3
			 */
			do_action( 'mainwp_before_jetpack_protect_overview_widgets' );


						?>
			<div class="gridster" id="mainwp-grid-wrapper">
				<?php do_action( 'mainwp_do_widget_boxes', $screen ); ?>
	</div>
		<?php
		/**
		 * Action: 'mainwp_after_jetpack_protect_overview_widgets'
		 *
		 * Fires at the bottom of the Overview page (after the last widget).
		 *
		 * @since 4.3
		 */
		do_action( 'mainwp_after_jetpack_protect_overview_widgets' );
		?>
	<script type="text/javascript">
		jQuery( document ).ready( function( $ ) {
			jQuery( '.mainwp-widget .mainwp-dropdown-tab .item' ).tab();
			mainwp_jetpack_protect_overview_screen_options = function () {
				jQuery( '#mainwp-jetpack-protect-overview-screen-options-modal' ).modal( {
					allowMultiple: true,
					onHide: function () {
					}
				} ).modal( 'show' );
				return false;
			};
			jQuery('#reset-jetpack-protect-overview-settings').on('click', function () {
				mainwp_confirm(__('Are you sure.'), function () {
				jQuery('input[name=number_overview_columns][value=2]').prop('checked', true);
				jQuery('.mainwp_hide_wpmenu_checkboxes input[name="mainwp_show_widgets[]"]').prop('checked', true);
				jQuery('input[name=reset_jetpack_protect_overview_settings]').attr('value', 1);
				jQuery('#submit-jetpack-protect-overview-settings').click();
				}, false, false, true);
				return false;
			});

		} );
	</script>

	<div class="ui modal" id="mainwp-jetpack-protect-overview-screen-options-modal">
		<i class="close icon"></i>
			<div class="header"><?php esc_html_e( 'Page Settings', 'mainwp-jetpack-protect-extension' ); ?></div>
			<div class="content ui form">
				<?php
				/**
				 * Action: mainwp_jetpack_protect_overview_screen_options_top
				 *
				 * Fires at the top of the Sceen Options modal on the Overview page.
				 */
				do_action( 'mainwp_jetpack_protect_overview_screen_options_top' );
				?>
				<form method="POST" action="" name="mainwp_jetpack_protect_overview_screen_options_form" id="mainwp-jetpack-protect-overview-screen-options-form">
					<?php wp_nonce_field( 'mainwp-admin-nonce' ); ?>
					<input type="hidden" name="mainwp_jetpack_protect_screen_opts_nonce" value="<?php echo wp_create_nonce( 'mainwp_jetpack_protect_screen_opts_nonce' ); ?>" />
					<?php echo self::render_screen_options(); ?>
					<?php
					/**
					 * Action: mainwp_jetpack_protect_overview_screen_options_bottom
					 *
					 * Fires at the bottom of the Sceen Options modal on the Overview page.
					 */
					do_action( 'mainwp_jetpack_protect_overview_screen_options_bottom' );
					?>
			</div>
			<div class="actions">
				<div class="ui two columns grid">
					<div class="left aligned column">
						<span data-tooltip="<?php esc_attr_e( 'Returns this page to the state it was in when installed. The feature also restores any widgets you have moved through the drag and drop feature on the page.', 'mainwp-jetpack-protect-extension' ); ?>" data-inverted="" data-position="top left"><input type="button" class="ui button" name="reset" id="reset-jetpack-protect-overview-settings" value="<?php esc_attr_e( 'Reset Page', 'mainwp-jetpack-protect-extension' ); ?>" /></span>
					</div>
					<div class="ui right aligned column">
						<input type="submit" class="ui green button" id="submit-jetpack-protect-overview-settings" value="<?php esc_attr_e( 'Save Settings', 'mainwp-jetpack-protect-extension' ); ?>" />
					</div>
				</div>
			</div>

			<input type="hidden" name="reset_jetpack_protect_overview_settings" value="" />
			</form>
		</div>

		<?php
		do_action( 'mainwp_render_modal_upload_icon' ); // render the modal.
	}

	/**
	 * Set the number of page coumns.
	 *
	 * @param mixed $columns Number of Columns.
	 * @param mixed $screen Screen size.
	 *
	 * @return int $columns Number of desired page columns.
	 */
	public function hook_screen_layout_columns( $columns, $screen ) {
		if ( $screen == self::$page ) {
			$columns[ self::$page ] = 2;
		}
		return $columns;
	}

	/**
	 * Method render_screen_options()
	 *
	 * Render Page Settings.
	 *
	 * @return void  Render Page Settings html.
	 */
	public static function render_screen_options() { // phpcs:ignore -- Current complexity is the only way to achieve desired results, pull request solutions appreciated.

		$default_widgets = array(
			'jetpack-protect'         => __( 'Jetpack Protect', 'mainwp-jetpack-protect-extension' ),
			'jetpack-protect-details' => __( 'Jetpack Protect details', 'mainwp-jetpack-protect-extension' ),
		);

		$custom_opts = array();
		/**
		 * Filter: mainwp_jetpack_protect_widgets_screen_options
		 *
		 * Filters available widgets on the Overview page allowing users to unsent unwanted widgets.
		 */
		$custom_opts = apply_filters( 'mainwp_jetpack_protect_widgets_screen_options', $custom_opts );

		if ( is_array( $custom_opts ) && 0 < count( $custom_opts ) ) {
			$default_widgets = array_merge( $default_widgets, $custom_opts );
		}

		$show_widgets = get_user_option( 'mainwp_jetpack_protect_show_widgets' );
		if ( ! is_array( $show_widgets ) ) {
			$show_widgets = array();
		}

		/**
		 * Action: mainwp_jetpack_protect_screen_options_modal_top
		 *
		 * Fires at the top of the Page Settings modal element.
		 *
		 * @since 4.1
		 */
		do_action( 'mainwp_jetpack_protect_screen_options_modal_top' );
		?>
		<?php if ( isset( $_GET['page'] ) ) : ?>
			<?php
			$overviewColumns = get_user_option( 'mainwp_number_jetpack_protect_overview_columns' );
			if ( 2 != $overviewColumns && 3 != $overviewColumns ) {
				$overviewColumns = 2;
			}

			?>
		<div class="ui grid field">
			<label class="six wide column middle aligned"><?php esc_html_e( 'Widgets columns', 'mainwp-jetpack-protect-extension' ); ?></label>
			<div class="ten wide column">
				<div class="ui radio checkbox">
					<input type="radio" name="number_overview_columns" required="required" <?php echo ( 2 == $overviewColumns ? 'checked="true"' : '' ); ?> value="2">
					<label><?php esc_html_e( 'Show widgets in 2 columns', 'mainwp-jetpack-protect-extension' ); ?></label>
				</div>
					<div class="ui fitted hidden divider"></div>
				<div class="ui radio checkbox">
					<input type="radio" name="number_overview_columns" required="required" <?php echo ( 3 == $overviewColumns ? 'checked="true"' : '' ); ?> value="3">
					<label><?php esc_html_e( 'Show widgets in 3 columns', 'mainwp-jetpack-protect-extension' ); ?></label>
				</div>
			</div>
		</div>

		<div class="ui grid field">
			<label class="six wide column"><?php esc_html_e( 'Show widgets', 'mainwp-jetpack-protect-extension' ); ?></label>
			<div class="ten wide column" <?php echo 'data-tooltip="' . esc_attr_e( 'Select widgets that you want to hide in the MainWP Overview page.', 'mainwp-jetpack-protect-extension' ); ?> data-inverted="" data-position="top left">
				<ul class="mainwp_hide_wpmenu_checkboxes">
				<?php
				foreach ( $default_widgets as $name => $title ) {
					$_selected = '';
					if ( ! isset( $show_widgets[ $name ] ) || 1 == $show_widgets[ $name ] ) {
						$_selected = 'checked';
					}
					?>
					<li>
						<div class="ui checkbox">
							<input type="checkbox" id="mainwp_show_widget_<?php echo esc_attr( $name ); ?>" name="mainwp_show_widgets[]" <?php echo $_selected; ?> value="<?php echo esc_attr( $name ); ?>">
							<label for="mainwp_show_widget_<?php echo esc_attr( $name ); ?>" ><?php echo esc_html( $title ); ?></label>
						</div>
						<input type="hidden" name="mainwp_widgets_name[]" value="<?php echo esc_attr( $name ); ?>">
					</li>
					<?php
				}
				?>
				</ul>
			</div>
		</div>
		<?php endif; ?>
		<?php
		/**
		 * Action: mainwp_jetpack_protect_screen_options_modal_bottom
		 *
		 * Fires at the bottom of the Page Settings modal element.
		 *
		 * @since 4.1
		 */
		do_action( 'mainwp_jetpack_protect_screen_options_modal_bottom' );
	}


}
