<?php
/**
 * MainWP Jetpack Protect Utility
 *
 * This file handles Plugin utility.
 *
 * @package MainWP/Extensions
 */

namespace MainWP\Extensions\JetpackProtect;

/**
 * Class MainWP_Jetpack_Protect_Utility
 */
class MainWP_Jetpack_Protect_Utility {

	/**
	 * Protected static variable to hold the single instance of the class.
	 *
	 * @var mixed Default null
	 */
	public static $instance = null;

	protected $option_handle = 'mainwp_jetpack_protect_options';


	protected $option;

	/**
	 * Method get_instance()
	 *
	 * Create a new public static instance of
	 * MainWP_Jetpack_Protect_Utility().
	 *
	 * @return void $instance New public static Instance.
	 */
	static function get_instance() {
		if ( null == self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * MainWP_Jetpack_Protect_Utility class constructor.
	 *
	 * @return void
	 */
	public function __construct() {
		$this->option = get_option( $this->option_handle );
	}

	/**
	 * Get Option.
	 *
	 * @param string $key Option key.
	 * @param mixed  $default default value.
	 *
	 * @return mixed Retruns option value.
	 */
	public function get_option( $key = null, $default = '' ) {
		if ( isset( $this->option[ $key ] ) ) {
			return $this->option[ $key ];
		}
		return $default;
	}

	/**
	 * Set Option
	 *
	 * Sets option in Databse.
	 *
	 * @param string $key Option key.
	 * @param mixed  $value Option value.
	 *
	 * @return mixed Update option.
	 */
	public function set_option( $key, $value ) {
		$this->option[ $key ] = $value;
		return update_option( $this->option_handle, $this->option );
	}

	/**
	 * Get timestamp.
	 *
	 * @param string $timestamp Holds Timestamp.
	 *
	 * @return float|int Return GMT offset.
	 */
	public static function get_timestamp( $timestamp = false ) {
		if ( empty( $timestamp ) ) {
			$timestamp = time();
		}
		$gmtOffset = get_option( 'gmt_offset' );

		return ( $gmtOffset ? ( $gmtOffset * HOUR_IN_SECONDS ) + $timestamp : $timestamp );
	}

	/**
	 * Format timestamp.
	 *
	 * @param string $timestamp Holds Timestamp.
	 * @param bool   $gmt Whether to set as General mountain time. Default: FALSE.
	 *
	 * @return string Return Timestamp.
	 */
	public static function format_timestamp( $timestamp, $gmt = false ) {
		return date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp, $gmt );
	}

	/**
	 * Format datestamp.
	 *
	 * @param string $timestamp Holds Timestamp.
	 * @param bool   $gmt Whether to set as General mountain time. Default: FALSE.
	 *
	 * @return string Return Timestamp.
	 */
	public static function format_datestamp( $timestamp, $gmt = false ) {
		return date_i18n( get_option( 'date_format' ), $timestamp, $gmt );
	}

	/**
	 * Method map_fields()
	 *
	 * Map Site.
	 *
	 * @param mixed $website Website to map.
	 * @param mixed $keys Keys to map.
	 * @param bool  $object_output Output format array|object.
	 *
	 * @return object $outputSite Mapped site.
	 */
	public static function map_fields( &$website, $keys, $object_output = false ) {
		$outputSite = array();
		if ( ! empty( $website ) ) {
			if ( is_object( $website ) ) {
				foreach ( $keys as $key ) {
					if ( property_exists( $website, $key ) ) {
						$outputSite[ $key ] = $website->$key;
					}
				}
			} elseif ( is_array( $website ) ) {
				foreach ( $keys as $key ) {
					$outputSite[ $key ] = $website[ $key ];
				}
			}
		}

		if ( $object_output ) {
			return (object) $outputSite;
		} else {
			return $outputSite;
		}
	}

	/**
	 * Method get_vulnerabilities_number_color()
	 *
	 * Get_vulnerabilities number color.
	 *
	 * @return string color.
	 */
	public static function get_vulnerabilities_number_color( $num_vuln ) {
		$v_color = '';
		if ( false !== $num_vuln ) {
			if ( 0 < $num_vuln ) {
				$v_color = 'red';
			} else {
				$v_color = 'green';
			}
		}
		return $v_color;
	}


	/**
	 * Debugging log info.
	 *
	 * Sets logging for debugging purpose.
	 *
	 * @param string $message Log info message.
	 */
	public static function log_info( $message ) {
		// Set color: 0 - LOG, 1 - WARNING, 2 - INFO, 3- DEBUG.
		$log_color = 2;
		do_action( 'mainwp_log_action', 'Jetpack Protect :: ' . $message, MAINWP_JETPACK_PROTECT_LOG_PRIORITY, $log_color );
	}

	/**
	 * Debugging log.
	 *
	 * Sets logging for debugging purpose.
	 *
	 * @param string $message Log debug message.
	 */
	public static function log_debug( $message ) {
		// Set color: 0 - LOG, 1 - WARNING, 2 - INFO, 3- DEBUG.
		$log_color = 3;
		do_action( 'mainwp_log_action', 'Jetpack Protect:: ' . $message, MAINWP_JETPACK_PROTECT_LOG_PRIORITY, $log_color );
	}

	 /**
	  * Show Info Messages
	  *
	  * Check whenther or not to show the MainWP Message.
	  *
	  * @param string $notice_id Notice ID.
	  *
	  * @return bool False if hidden, true to show.
	  */
	public static function show_mainwp_message( $notice_id ) {
		$status = get_user_option( 'mainwp_notice_saved_status' );
		if ( ! is_array( $status ) ) {
			$status = array();
		}
		if ( isset( $status[ $notice_id ] ) ) {
			return false;
		}
		return true;
	}

	/**
	 * Method get_page_id()
	 *
	 * Get current Page ID.
	 *
	 * @param null $screen Current Screen ID.
	 *
	 * @return string $page Current page ID.
	 */
	public static function get_page_id( $screen = null ) {

		if ( empty( $screen ) ) {
			$screen = get_current_screen();
		} elseif ( is_string( $screen ) ) {
			$screen = convert_to_screen( $screen );
		}

		if ( ! isset( $screen->id ) ) {
			return;
		}

		$page = $screen->id;

		return $page;
	}

}
