<?php
/*
 * Plugin Name: MainWP Jetpack Protect Extension
 * Plugin URI: https://mainwp.com
 * Description: MainWP Jetpack Protect Extension uses the Jetpack Protect plugin to bring you information about vulnerable plugins and themes on your Child Sites so you can act accordingly.
 * Version: 5.0.4
 * Author: MainWP
 * Author URI: https://mainwp.com
 * Documentation URI: https://mainwp.com/help/docs/category/mainwp-extensions/jetpack-protect/
 */

namespace MainWP\Extensions\JetpackProtect;

if ( ! defined( 'MAINWP_JETPACK_PROTECT_PLUGIN_URL' ) ) {
	define( 'MAINWP_JETPACK_PROTECT_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}

if ( ! defined( 'MAINWP_JETPACK_PROTECT_LOG_PRIORITY' ) ) {
	define( 'MAINWP_JETPACK_PROTECT_LOG_PRIORITY', 20240916 );
}

/**
 * Class MainWP_Virusdie_Extension_Activator
 */
class MainWP_Jetpack_Protect_Extension_Activator {
	/**
	 * Protected variable containg information about MainWP plugin status.
	 *
	 * @var bool
	 */
	protected $mainwpMainActivated = false;

	/**
	 * Protected variable containg information about the Extension status.
	 *
	 * @var bool
	 */
	protected $childEnabled = false;

	/**
	 * Protected variable containg the Extension key.
	 *
	 * @var bool|string
	 */
	protected $childKey = false;

	/**
	 * Protected variable containg extension file.
	 *
	 * @var string
	 */
	protected $childFile;

	/**
	 * Protected variable containg the extension handle.
	 *
	 * @var string
	 */
	protected $plugin_handle = 'mainwp-jetpack-protect-extension';

	/**
	 * Protected variable containg the extension ID (product title).
	 *
	 * @var string
	 */
	protected $product_id = 'MainWP Jetpack Protect Extension';

	/**
	 * Protected variable containg the extension version number.
	 *
	 * @var string
	 */
	protected $software_version = '5.0.4';

	/**
	 * MainWP_Jetpack_Protect_Extension_Activator class constructor.
	 *
	 * @uses register_activation_hook() Set the activation hook for a plugin.
	 * @see https://developer.wordpress.org/reference/functions/register_activation_hook/
	 *
	 * @uses register_deactivation_hook() Set the deactivation hook for a plugin.
	 * @see https://developer.wordpress.org/reference/functions/register_deactivation_hook/
	 *
	 * @return void
	 */
	public function __construct() {
		$this->childFile = __FILE__;

		spl_autoload_register( array( $this, 'autoload' ) );
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );

		add_filter( 'mainwp_getextensions', array( &$this, 'get_this_extension' ) );

		add_filter( 'mainwp_sync_extensions_options', array( &$this, 'hook_sync_extensions_options' ), 10, 1 );
		add_action( 'mainwp_applypluginsettings_mainwp-jetpack-protect-extension', array( &$this, 'hook_apply_plugin_settings' ) );
		add_filter( 'mainwp_sync_others_data', array( MainWP_Jetpack_Protect_Extension::get_instance(), 'hook_sync_others_data' ), 10, 2 );
		add_action( 'mainwp_site_synced', array( MainWP_Jetpack_Protect_Extension::get_instance(), 'hook_site_synced' ), 10, 2 );
		add_filter( 'mainwp_log_specific_actions', array( $this, 'hook_log_specific_actions' ), 10, 2 );

		$this->mainwpMainActivated = apply_filters( 'mainwp_activated_check', false );
		if ( $this->mainwpMainActivated !== false ) {
			$this->activate_this_plugin();
		} else {
			add_action( 'mainwp_activated', array( &$this, 'activate_this_plugin' ) );
		}

		add_action( 'admin_notices', array( &$this, 'extension_notice' ) );

		// includes rest api work.
		require_once 'class/class-mainwp-jetpack-protect-rest-api.php';
		Rest_Api::instance()->init();
	}

	/**
	 * Hook sync extension options for new site.
	 *
	 * @return void
	 */
	public function hook_sync_extensions_options( $values = array() ) {
		$values['mainwp-jetpack-protect-extension'] = array(
			'plugin_slug'          => 'jetpack-protect/jetpack-protect.php',
			'plugin_name'          => 'Jetpack Protect',
			'no_setting'           => true,
			'action_after_install' => esc_html__( 'Connect Jetpack Protect', 'mainwp-jetpack-protect-extension' ),
		);
		return $values;
	}


	/**
	 * Hook hook_log_specific_actions.
	 *
	 * Hook sites table sortable columns.
	 *
	 * @return mixed $sortable_columns.
	 */
	public function hook_log_specific_actions( $inputs ) {
		$inputs[ MAINWP_JETPACK_PROTECT_LOG_PRIORITY ] = __( 'Jetpack Protect', 'mainwp-jetpack-protect-extension' );
		return $inputs;
	}

	/**
	 * Method hook_apply_plugin_settings().
	 *
	 * @param int $siteid site id.
	 *
	 * @return void
	 */
	public function hook_apply_plugin_settings( $siteid = false ) {
		$information = MainWP_Jetpack_Protect::get_instance()->handle_connect_disconnect( $siteid, 'connect' );

		$result = array();
		if ( is_array( $information ) ) {
			if ( isset( $information['code'] ) && 'success' === $information['code'] ) {
				$result = array( 'result' => 'success' );
			} elseif ( $information['error'] ) {
				$result = array( 'error' => $information['error'] );
			} else {
				$result = array( 'result' => 'failed' );
			}
		} else {
			$result = array( 'result' => 'failed' );
		}
		die( json_encode( $result ) );
	}

	/**
	 * Loads class automatically.
	 *
	 * @param string $class_name Class name.
	 *
	 * @return void
	 */
	public function autoload( $class_name ) {

		if ( 0 === strpos( $class_name, 'MainWP\Extensions\JetpackProtect' ) ) {
			// trim the namespace prefix: MainWP\Extensions\JetpackProtect\.
			$class_name = str_replace( 'MainWP\Extensions\JetpackProtect\\', '', $class_name );
		} else {
			return;
		}

		if ( 0 !== strpos( $class_name, 'MainWP_Jetpack_Protect' ) ) {
			return;
		}

		$class_name = strtolower( str_replace( '_', '-', $class_name ) );

		$class_file = WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . str_replace( basename( __FILE__ ), '', plugin_basename( __FILE__ ) ) . 'class' . DIRECTORY_SEPARATOR . 'class-' . $class_name . '.php';
		if ( file_exists( $class_file ) ) {
			require_once $class_file;
		}
	}


	/**
	 * Add your extension to MainWP via the 'mainwp_getextensions' filter.
	 *
	 * @param array $params Array containing the extensions info.
	 *
	 * @return array $params Updated array containing the extensions info.
	 */
	public function get_this_extension( $pArray ) {

		$pArray[] = array(
			'plugin'     => __FILE__,
			'api'        => $this->plugin_handle,
			'mainwp'     => true,
			'apiManager' => true,
			'callback'   => array( $this, 'settings' ),
		);

		return $pArray;
	}


	/**
	 * Displays the extension page with adequate header and footer.
	 *
	 * @uses MainWP_Virusdie::render_pages()
	 */
	public function settings() {
		do_action( 'mainwp_pageheader_extensions', __FILE__ );
		MainWP_Jetpack_Protect::render();
		do_action( 'mainwp_pagefooter_extensions', __FILE__ );
	}

	/**
	 * Activate the extension API license and initiate the extension.
	 */
	public function activate_this_plugin() {
		$this->mainwpMainActivated = apply_filters( 'mainwp_activated_check', $this->mainwpMainActivated );
		$this->childEnabled        = apply_filters( 'mainwp_extension_enabled_check', __FILE__ );
		$this->childKey            = $this->childEnabled['key'];

		if ( function_exists( 'mainwp_current_user_can' ) && ! mainwp_current_user_can( 'extension', 'mainwp-jetpack-protect-extension' ) ) {
			return;
		}

		add_filter( 'mainwp_getsubpages_sites', array( MainWP_Jetpack_Protect_Overview::get_instance(), 'hook_managesites_subpage' ), 10, 1 );
		add_filter( 'mainwp_getmetaboxes', array( &$this, 'hook_get_metaboxes' ) );
		add_filter( 'mainwp_widgets_screen_options', array( &$this, 'hook_widgets_screen_options' ), 10, 1 );

		MainWP_Jetpack_Protect_Extension::get_instance();
	}

	/**
	 * Adds metabox (widget) on the MainWP Dashboard overview page via the 'mainwp_getmetaboxes' filter.
	 *
	 * @param array $metaboxes Array containing metaboxes data.
	 *
	 * @return array $metaboxes Updated array that contains metaboxes data.
	 */
	public function hook_get_metaboxes( $metaboxes ) {
		if ( ! $this->childEnabled ) {
			return $metaboxes;
		}

		if ( ! is_array( $metaboxes ) ) {
			$metaboxes = array();
		}
		$metaboxes[] = array(
			'id'            => 'jetpack-protect-widget',
			'plugin'        => $this->childFile,
			'key'           => $this->childKey,
			'metabox_title' => __( 'Jetpack Protect', 'mainwp-jetpack-protect-extension' ),
			'callback'      => array( MainWP_Jetpack_Protect_Overview::class, 'render_metabox' ),
			'icon'          => plugin_dir_url( __FILE__ ) . 'images/jetpack-protect.png',
			'name'          => 'Jetpack Protect',
		);
		return $metaboxes;
	}


	/**
	 * Widgets screen options.
	 *
	 * @param array $input Input.
	 *
	 * @return array $input Output.
	 */
	public function hook_widgets_screen_options( $input ) {
		$input['advanced-jetpack-protect-widget'] = __( 'Jetpack Protect', 'mainwp-jetpack-protect-extension' );
		return $input;
	}

	/**
	 * Get the extenion key.
	 *
	 * @return string
	 */
	public function get_child_key() {
		return $this->childKey;
	}

	/**
	 * Get the extension file.
	 *
	 * @return string
	 */
	public function get_child_file() {
		return $this->childFile;
	}

	/**
	 * Render the warning notice if the MainWP Dashboard plugin is not activated.
	 */
	public function extension_notice() {
		global $current_screen;
		if ( $current_screen->parent_base == 'plugins' && $this->mainwpMainActivated == false ) {
			echo '<div class="error"><p>' . sprintf( esc_html__( 'MainWP Jetpack Protect Extension requires %1$sMainWP Dashboard Plugin%2$s to be activated in order to work. Please install and activate %3$sMainWP Dashboard Plugin%4$s first.' ), '<a href="http://mainwp.com/" target="_blank">', '</a>', '<a href="http://mainwp.com/" target="_blank">', '</a>' ) . '</p></div>';
		}
	}

	/**
	 * Activate the extension info.
	 */
	public function activate() {
		$options = array(
			'product_id'       => $this->product_id,
			'software_version' => $this->software_version,
		);
		do_action( 'mainwp_activate_extention', $this->plugin_handle, $options );
	}
	/**
	 * Deactivate the extension info.
	 */
	public function deactivate() {
		do_action( 'mainwp_deactivate_extention', $this->plugin_handle );
	}
}

global $mainJetpackProtectExtensionActivator;
$mainJetpackProtectExtensionActivator = new MainWP_Jetpack_Protect_Extension_Activator();
