<?php
/*
 * Plugin Name: MainWP Vulnerability Checker Extension
 * Plugin URI: https://mainwp.com
 * Description: MainWP Vulnerability Checker extension brings you information about vulnerable plugins on your Child Sites so you can act accordingly.
 * Version: 5.0.3
 * Author: MainWP
 * Author URI: https://mainwp.com
 * Documentation URI: https://mainwp.com/help/docs/category/mainwp-extensions/vulnerability-checker/
 */

if ( ! defined( 'ABSPATH' ) ) {
	die( 'No direct access allowed' );
}

if ( ! defined( 'MAINWP_VULNER_CHECKER_PLUGIN_URL' ) ) {
	define( 'MAINWP_VULNER_CHECKER_PLUGIN_URL', plugins_url( '', __FILE__ ) );
}

if ( ! defined( 'MAINWP_VULNER_CHECKER_PLUGIN_SLUG' ) ) {
	define( 'MAINWP_VULNER_CHECKER_PLUGIN_SLUG', plugin_basename( __FILE__ ) );
}

if ( ! defined( 'MAINWP_VULNER_CHECKER_LOG_PRIORITY' ) ) {
	define( 'MAINWP_VULNER_CHECKER_LOG_PRIORITY', 70 );
}

class MainWP_Vulnerability_Checker_Extension { // phpcs:ignore -- NOSONAR - multi methods.

	public static $instance = null;

	public static function instance() {
		if ( null == self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	public function __construct() {
		add_action( 'admin_init', array( &$this, 'admin_init' ) );
		add_action( 'init', array( &$this, 'localization' ) );
		add_filter( 'mainwp_getsubpages_sites', array( &$this, 'managesites_subpage' ), 10, 1 );
		add_filter( 'mainwp_log_specific_actions', array( $this, 'hook_log_specific_actions' ), 10, 2 );
		MainWP_Vulnerability_DB::instance()->install();
		MainWP_Vulnerability_Checker::instance();
	}

	public function localization() {
		load_plugin_textdomain( 'mainwp-vulnerability-checker-extension', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
	}

	public function managesites_subpage( $subPage ) {
		$subPage[] = array(
			'title'       => __( 'Vulnerabilities', 'mainwp-vulnerability-checker-extension' ),
			'slug'        => 'Vulnerabilities',
			'sitetab'     => true,
			'menu_hidden' => true,
			'callback'    => array( 'MainWP_Vulnerability_Checker', 'render_detail' ),
		);
		return $subPage;
	}

	public static function is_premium_account() {
		$premium = true; // always.
		return $premium;
	}

	/**
	 * Hook hook_log_specific_actions.
	 *
	 * Hook sites table sortable columns.
	 *
	 * @return mixed $sortable_columns.
	 */
	public function hook_log_specific_actions( $inputs ) {
		$inputs[ MAINWP_VULNER_CHECKER_LOG_PRIORITY ] = __( 'Vuln Checker', 'mainwp-vulnerability-checker-extension' );
		return $inputs;
	}

	public function admin_init() {
		wp_enqueue_style( 'mainwp-vulner-checker-extension', MAINWP_VULNER_CHECKER_PLUGIN_URL . '/css/mainwp-vulner.css' );
		if ( isset( $_REQUEST['page'] ) && ( 'Extensions-Mainwp-Vulnerability-Checker-Extension' == $_REQUEST['page'] || 'ManageSitesVulnerabilities' == $_REQUEST['page'] ) ) {
			wp_enqueue_script( 'mainwp-vulner-checker-extension', MAINWP_VULNER_CHECKER_PLUGIN_URL . '/js/mainwp-vulner.js', array(), '1.1' );
		}
		wp_localize_script( 'mainwp-vulner-checker-extension', 'mainwp_vulner_loca', array( 'nonce' => wp_create_nonce( '_wpnonce_vulner' ) ) );
		if ( isset( $_GET['do'] ) && $_GET['do'] == 'cronVulnerCheck' ) {
			MainWP_Vulnerability_Checker::cron_vulner_recheck();
		}
	}
}


class MainWP_Vulnerability_Checker_Extension_Activator { // phpcs:ignore -- NOSONAR - multi methods.
	protected $mainwpMainActivated = false;
	protected $childEnabled        = false;
	protected $childKey            = false;
	protected $childFile;
	protected $plugin_handle    = 'mainwp-vulnerability-checker-extension';
	protected $product_id       = 'MainWP Vulnerability Checker Extension';
	protected $software_version = '5.0.3';
	public $mainwpVulner        = null;

	public function __construct() {

		$this->childFile = __FILE__;

		spl_autoload_register( array( $this, 'autoload' ) );
		register_activation_hook( __FILE__, array( $this, 'activate' ) );
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );

		add_filter( 'mainwp_getextensions', array( &$this, 'get_this_extension' ) );
		add_filter( 'mainwp_vulnerable_get_data', array( MainWP_Vulnerability_Hooks::instance(), 'hook_vulnerable_get_data' ), 10, 4 );

		$this->mainwpMainActivated = apply_filters( 'mainwp_activated_check', false );

		if ( $this->mainwpMainActivated !== false ) {
			$this->activate_this_plugin();
		} else {
			add_action( 'mainwp_activated', array( &$this, 'activate_this_plugin' ) );
		}
		add_action( 'admin_init', array( &$this, 'admin_init' ) );
		add_action( 'admin_notices', array( &$this, 'mainwp_error_notice' ) );
	}

	public function autoload( $class_name ) {
		$autoload_types = array( 'class' );
		foreach ( $autoload_types as $type ) {
			$autoload_dir  = \trailingslashit( __DIR__ . DIRECTORY_SEPARATOR . $type );
			$autoload_path = sprintf( '%s%s-%s.php', $autoload_dir, $type, strtolower( str_replace( '_', '-', $class_name ) ) );
			if ( file_exists( $autoload_path ) ) {
				require_once $autoload_path;
			}
		}
	}

	public function admin_init() {
		MainWP_Vulnerability_Checker::instance()->admin_init();
		MainWP_Vulnerability_Nvd_Checker::instance()->admin_init();
		MainWP_Vulnerability_Hooks::instance()->admin_init();
	}

	public function get_this_extension( $pArray ) {
		$pArray[] = array(
			'plugin'           => __FILE__,
			'api'              => $this->plugin_handle,
			'mainwp'           => true,
			'callback'         => array( &$this, 'settings' ),
			'apiManager'       => true,
			'on_load_callback' => '',
		);
		return $pArray;
	}

	public function settings() {
		do_action( 'mainwp_pageheader_extensions', __FILE__ );
		if ( $this->childEnabled ) {
			MainWP_Vulnerability_Checker::instance()->render();
		}
		do_action( 'mainwp_pagefooter_extensions', __FILE__ );
	}

	public function activate_this_plugin() {
		$this->mainwpMainActivated = apply_filters( 'mainwp_activated_check', $this->mainwpMainActivated );
		$this->childEnabled        = apply_filters( 'mainwp_extension_enabled_check', __FILE__ );
		$this->childKey            = $this->childEnabled['key'];
		if ( function_exists( 'mainwp_current_user_can' ) && ! mainwp_current_user_can( 'extension', 'mainwp-vulnerability-checker-extension' ) ) {
			return;
		}
		$this->mainwpVulner = new MainWP_Vulnerability_Checker_Extension();
	}

	public function get_metaboxes( $metaboxes ) {
		if ( ! is_array( $metaboxes ) ) {
			$metaboxes = array();
		}
		$metaboxes[] = array(
			'plugin'        => $this->childFile,
			'key'           => $this->childKey,
			'metabox_title' => 'Vulnerability Checker',
			'callback'      => array( 'MainWP_Vulnerability_Checker', 'render_metabox' ),
		);
		return $metaboxes;
	}

	public function get_child_key() {
		return $this->childKey;
	}

	public function get_child_file() {
		return $this->childFile;
	}

	public function mainwp_error_notice() {
		global $current_screen;
		if ( $current_screen->parent_base == 'plugins' && $this->mainwpMainActivated == false ) {
			echo '<div class="error"><p>MainWP Vulnerability Checker Extension ' . esc_html__( 'requires <a href="https://mainwp.com/" target="_blank" rel="noopener noreferrer">MainWP Dashboard Plugin</a> to be activated in order to work. Please install and activate <a href="https://mainwp.com/" target="_blank" rel="noopener noreferrer">MainWP Dashboard Plugin</a> first.' ) . '</p></div>';
		}
	}
	
	public function activate() {
		$options = array(
			'product_id'       => $this->product_id,
			'software_version' => $this->software_version,
		);
		do_action( 'mainwp_activate_extention', $this->plugin_handle, $options );
	}

	public function deactivate() {
		do_action( 'mainwp_deactivate_extention', $this->plugin_handle );
	}
}

global $mainwp_VulnerCheckerExtensionActivator;

$mainwp_VulnerCheckerExtensionActivator = new MainWP_Vulnerability_Checker_Extension_Activator();
