<?php
/**
 * Admin class for Security Ninja MainWP extension
 *
 * @package Security_Ninja_MainWP
 */

namespace WPSecurityNinja\MainWPAddon;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Security_Ninja_MainWP_Admin {

	public function __construct() {
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'admin_init', array( $this, 'handle_form_submissions' ) );
	}

	/**
	 * Enqueue admin scripts and styles
	 *
	 * @author  Lars Koudal
	 * @since   v0.0.1
	 * @version v1.0.0  Monday, May 5th, 2025.
	 * @access  public
	 * @return  void
	 */
	public function enqueue_scripts() {
		// Only enqueue on our extension page
		$screen = get_current_screen();
		if ( ! $screen || strpos( $screen->id, 'Extensions-Security-Ninja-For-Mainwp' ) === false ) {
			return;
		}

		// Enqueue admin styles
		wp_enqueue_style(
			'security-ninja-mainwp-admin',
			plugins_url( '../../assets/css/admin.css', __FILE__ ),
			array(),
			'1.0.0'
		);
	}

	/**
	 * Handle form submissions with proper nonce verification
	 *
	 * @author  Lars Koudal
	 * @since   v1.0.1
	 * @version v1.0.1  Monday, May 5th, 2025.
	 * @access  public
	 * @return  void
	 */
	public function handle_form_submissions() {
		// Only process on our extension page
		if ( ! isset( $_GET['page'] ) || 'Extensions-Security-Ninja-For-Mainwp' !== $_GET['page'] ) {
			return;
		}

		// Check if form was submitted
		if ( isset( $_POST['security_ninja_mainwp_action'] ) ) {
			// Verify nonce
			if ( ! wp_verify_nonce( $_POST['security_ninja_mainwp_nonce'], 'security_ninja_mainwp_action' ) ) {
				wp_die( esc_html__( 'Security check failed. Please try again.', 'security-ninja-mainwp' ) );
			}

			// Check user capabilities
			if ( ! current_user_can( 'manage_options' ) ) {
				wp_die( esc_html__( 'You do not have sufficient permissions to perform this action.', 'security-ninja-mainwp' ) );
			}

			// Handle different actions
			$action = sanitize_text_field( $_POST['security_ninja_mainwp_action'] );
			switch ( $action ) {
				case 'save_settings':
					$this->save_settings();
					break;
				// Add more actions as needed
			}
		}
	}

	/**
	 * Save settings with proper validation
	 *
	 * @author  Lars Koudal
	 * @since   v1.0.1
	 * @version v1.0.1  Monday, May 5th, 2025.
	 * @access  private
	 * @return  void
	 */
	private function save_settings() {
		// Validate and sanitize settings
		$settings = array();
		
		// Example: Save a setting if provided
		if ( isset( $_POST['security_ninja_setting'] ) ) {
			$settings['security_ninja_setting'] = sanitize_text_field( $_POST['security_ninja_setting'] );
		}

		// Save to database
		update_option( 'security_ninja_mainwp_settings', $settings );

		// Add success message
		add_action( 'admin_notices', function() {
			echo '<div class="notice notice-success is-dismissible"><p>' . 
				 esc_html__( 'Settings saved successfully.', 'security-ninja-mainwp' ) . 
				 '</p></div>';
		});
	}

	/**
	 * Render the main page
	 *
	 * @author  Lars Koudal
	 * @since   v0.0.1
	 * @version v1.0.0  Monday, May 5th, 2025.
	 * @access  public
	 * @return  void
	 */
	public function render_page() {
		// Double check capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'security-ninja-mainwp' ) );
		}

		// Get current tab
		$current_tab = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : 'dashboard';

		// MainWP UI wrapper
		echo '<div class="ui segment">';
		$this->render_tabs( $current_tab );
		$this->render_tab_content( $current_tab );
		echo '</div>';
	}

	/**
	 * Render the tabs
	 *
	 * @author  Lars Koudal
	 * @since   v0.0.1
	 * @version v1.0.0  Monday, May 5th, 2025.
	 * @access  private
	 * @param   mixed   $current_tab
	 * @return  void
	 */
	private function render_tabs( $current_tab ) {
		$tabs = array(
			'dashboard' => __( 'All Events', 'security-ninja-mainwp' ),
			'settings'  => __( 'Settings', 'security-ninja-mainwp' ),
		);

		echo '<div class="ui secondary pointing menu">';
		foreach ( $tabs as $tab_id => $tab_name ) {
			$active = $current_tab === $tab_id ? 'active' : '';
			$url    = add_query_arg(
				array(
					'page' => 'Extensions-Security-Ninja-For-Mainwp',
					'tab'  => $tab_id,
				),
				admin_url( 'admin.php' )
			);
			echo '<a href="' . esc_url( $url ) . '" class="item ' . esc_attr( $active ) . '">' . esc_html( $tab_name ) . '</a>';
		}
		echo '</div>';
	}

	/**
	 * Render the tab content
	 */
	private function render_tab_content( $current_tab ) {
		echo '<div class="ui tab segment active">';
		switch ( $current_tab ) {
			case 'dashboard':
				$this->render_dashboard_tab();
				break;
			case 'settings':
				$this->render_settings_tab();
				break;
			default:
				$this->render_dashboard_tab();
				break;
		}
		echo '</div>';
	}

	/**
	 * Render the dashboard tab
	 *
	 * @author  Lars Koudal
	 * @since   v0.0.1
	 * @version v1.0.0  Monday, May 5th, 2025.
	 * @access  private
	 * @return  void
	 */
	private function render_dashboard_tab() {
		global $snmwp_fs;
		$docs_link = \WPSecurityNinja\MainWPAddon\Security_Ninja_Mainwp::generate_sn_web_link( 'welcome_notice', '/docs/mainwp/' );

		$show_premium_message = true;

		if ( $snmwp_fs && $snmwp_fs->can_use_premium_code__premium_only() ) {
			$show_premium_message = false;
		}

		if ( $show_premium_message ) {
?>

		<div class="ui message">
			<div class="header">
				<?php esc_html_e( 'Security Ninja Events on all your sites', 'security-ninja-mainwp' ); ?>
			</div>
			<p><?php esc_html_e( 'Monitor all events and user activities from your sites in one place.', 'security-ninja-mainwp' ); ?> <?php esc_html_e( 'Only premium plan events data is displayed here.', 'security-ninja-mainwp' ); ?> <a href="<?php echo esc_url( $docs_link ); ?>" target="_blank" rel="noopener"><?php esc_html_e( 'Documentation', 'security-ninja-mainwp' ); ?></a>
			</p>
		</div>
<?php
		}
		\WPSecurityNinja\MainWPAddon\SecurityNinjaMainWPExtension::render_page();
	}

	/**
	 * Render the settings tab with proper form security
	 *
	 * @author  Lars Koudal
	 * @since   v0.0.1
	 * @version v1.0.1  Monday, May 5th, 2025.
	 * @access  private
	 * @return  void
	 */
	private function render_settings_tab() {
		$settings = get_option( 'security_ninja_mainwp_settings', array() );
		$current_setting = isset( $settings['security_ninja_setting'] ) ? $settings['security_ninja_setting'] : '';
		?>
		<div class="ui message">
			<div class="header">
				<?php esc_html_e( 'Settings', 'security-ninja-mainwp' ); ?>
			</div>
			<p><?php esc_html_e( 'No settings available yet.', 'security-ninja-mainwp' ); ?></p>
		</div>

		<?php
		// @TODO: Add settings form here
		/*
		<form method="post" action="" class="ui form">
			<?php wp_nonce_field( 'security_ninja_mainwp_action', 'security_ninja_mainwp_nonce' ); ?>
			<input type="hidden" name="security_ninja_mainwp_action" value="save_settings">
			
			<div class="field">
				<label for="security_ninja_setting"><?php esc_html_e( 'Example Setting', 'security-ninja-mainwp' ); ?></label>
				<input type="text" id="security_ninja_setting" name="security_ninja_setting" value="<?php echo esc_attr( $current_setting ); ?>">
			</div>

			<button type="submit" class="ui primary button">
				<?php esc_html_e( 'Save Settings', 'security-ninja-mainwp' ); ?>
			</button>
		</form>
		*/


	}
}
